/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.config;

import java.io.IOException;
import java.util.List;
import java.util.TimeZone;

import org.opennms.netmgt.config.syslogd.HideMatch;
import org.opennms.netmgt.config.syslogd.UeiMatch;

/**
 * <p>SyslogdConfig interface.</p>
 *
 * @author ranger
 * @version $Id: $
 */
public interface SyslogdConfig {
    /**
     * <p>getSyslogPort</p>
     *
     * @return a int.
     */
    int getSyslogPort();

    /**
     * <p>getListenAddress</p>
     *
     * @return a {@link java.lang.String} object.
     * @since 1.8.1
     */
    String getListenAddress();
    
    /**
     * <p>getNewSuspectOnMessage</p>
     *
     * @return a boolean.
     */
    boolean getNewSuspectOnMessage();

    /**
     * <p>getForwardingRegexp</p>
     *
     * @return a {@link java.lang.String} object.
     */
    String getForwardingRegexp();

    /**
     * <p>getMatchingGroupHost</p>
     *
     * @return a int.
     */
    Integer getMatchingGroupHost();

    /**
     * <p>getMatchingGroupMessage</p>
     *
     * @return a int.
     */
    Integer getMatchingGroupMessage();

    /**
     * <p>getParser</p>
     *
     * @return the parser class to use when parsing syslog messages, as a string.
     */
    String getParser();

    /**
     * A collection of Strings->UEI's
     */
    List<UeiMatch> getUeiList();

    /**
     * A collection of Strings we do not want to attach to the event.
     */
    List<HideMatch> getHideMessages();
    
    /**
     * <p>getDiscardUei</p>
     *
     * @return a {@link java.lang.String} object.
     */
    String getDiscardUei();

    /**
     * Number of threads used for consuming/dispatching messages.
     *
     * @return number of threads
     */
    int getNumThreads();

    /**
     * Maximum number of messages to keep in memory while waiting
     * to be dispatched.
     *
     * @return queue size
     */
    int getQueueSize();

    /**
     * Messages are aggregated in batches before being dispatched.
     *
     * When the batch reaches this size, it will be dispatched.
     *
     * @return batch size
     */
    int getBatchSize();

    /**
     * Messages are aggregated in batches before being dispatched.
     *
     * When the batch has been created for longer than this interval
     * it will be dispatched, regardless of the size.
     *
     * @return interval in ms
     */
    int getBatchIntervalMs();

    /**
     * Optional:
     * - if not null it will be used as default time zone if no time zone is given
     * - if not set the system time zone will be used
     */
    TimeZone getTimeZone();

    /**
     * Indicates whether or not to include the raw syslog message as an event parameter on events generated by Syslogd.
     * <p>
     * Default behavior is false unless configured otherwise.
     *
     * @return whether or not to include the raw syslog message as an event parameter on events generated by Syslogd
     */
    boolean shouldIncludeRawSyslogmessage();

    /**
     * reload configuration
     * @throws IOException
     */
    void reload() throws IOException;
}
