/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.features.vaadin.dashboard.dashlets;

import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;

import org.apache.commons.lang.CharEncoding;
import org.opennms.features.vaadin.dashboard.model.AbstractDashlet;
import org.opennms.features.vaadin.dashboard.model.AbstractDashletComponent;
import org.opennms.features.vaadin.dashboard.model.Dashlet;
import org.opennms.features.vaadin.dashboard.model.DashletComponent;
import org.opennms.features.vaadin.dashboard.model.DashletSpec;

import com.vaadin.server.ExternalResource;
import com.vaadin.ui.BrowserFrame;
import com.vaadin.ui.Component;
import com.vaadin.ui.UI;
import com.vaadin.ui.VerticalLayout;
import com.vaadin.v7.ui.Label;

/**
 * This class implements a {@link Dashlet} for displaying an Grafana dashboard.
 *
 * @author Christian Pape
 */
public class GrafanaDashlet extends AbstractDashlet {

    private DashletComponent m_dashletComponent;

    /**
     * Constructor for instantiating new objects.
     *
     * @param dashletSpec the {@link DashletSpec} to be used
     */
    public GrafanaDashlet(String name, DashletSpec dashletSpec) {
        super(name, dashletSpec);
    }

    @Override
    public DashletComponent getWallboardComponent(final UI ui) {
        if (m_dashletComponent == null) {
            m_dashletComponent = new AbstractDashletComponent() {
                private VerticalLayout m_verticalLayout = new VerticalLayout();

                {
                    m_verticalLayout.setCaption(getName());
                    m_verticalLayout.setSizeFull();
                }

                @Override
                public void refresh() {
                    m_verticalLayout.removeAllComponents();

                    String uri = "";
                    String from = "";
                    String to = "";

                    /**
                     * Retrieving the required parameters
                     */
                    if (getDashletSpec().getParameters().containsKey("uri")) {
                        uri = getDashletSpec().getParameters().get("uri");
                    }

                    if (getDashletSpec().getParameters().containsKey("from")) {
                        from = getDashletSpec().getParameters().get("from");
                    }

                    if (getDashletSpec().getParameters().containsKey("to")) {
                        to = getDashletSpec().getParameters().get("to");
                    }

                    if (!"".equals(uri)) {
                        /**
                         * Retrieving the properties
                         */
                        final String grafanaProtocol = System.getProperty("org.opennms.grafanaBox.protocol", "http");
                        final String grafanaHostname = System.getProperty("org.opennms.grafanaBox.hostname", "localhost");
                        final int grafanaPort = Integer.parseInt(System.getProperty("org.opennms.grafanaBox.port", "3000"));

                        try {
                            /**
                             * Constructing the URL...
                             */
                            URL url = new URL(
                                    String.format(
                                        "%s://%s:%d/dashboard/%s?from=%s&to=%s",
                                        grafanaProtocol,
                                        grafanaHostname,
                                        grafanaPort,
                                        uri,
                                        URLEncoder.encode(from, CharEncoding.UTF_8),
                                        URLEncoder.encode(to, CharEncoding.UTF_8)
                                    )
                            );

                            /**
                             * creating browser frame to display the URL
                             */
                            BrowserFrame browserFrame = new BrowserFrame(null, new ExternalResource(url));
                            browserFrame.setSizeFull();
                            m_verticalLayout.addComponent(browserFrame);
                        } catch (UnsupportedEncodingException e) {
                            e.printStackTrace();
                        } catch (MalformedURLException e) {
                            m_verticalLayout.addComponent(new Label("MalformedURLException: " + e.getMessage()));
                            return;
                        }
                    } else {
                        m_verticalLayout.addComponent(new Label("No URL specified!"));
                    }
                }

                @Override
                public Component getComponent() {
                    return m_verticalLayout;
                }
            };
        }

        return m_dashletComponent;
    }

    @Override
    public DashletComponent getDashboardComponent(final UI ui) {
        return getWallboardComponent(ui);
    }
}
