/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.telemetry.protocols.sflow.parser.proto.flows;

import java.util.Optional;

import org.bson.BsonWriter;
import org.opennms.netmgt.telemetry.listeners.utils.BufferUtils;
import org.opennms.netmgt.telemetry.protocols.sflow.parser.SampleDatagramEnrichment;
import org.opennms.netmgt.telemetry.protocols.sflow.parser.InvalidPacketException;
import org.opennms.netmgt.telemetry.protocols.sflow.parser.SampleDatagramVisitor;
import org.opennms.netmgt.telemetry.protocols.sflow.parser.proto.Array;

import com.google.common.base.MoreObjects;

import io.netty.buffer.ByteBuf;

// struct flow_sample_expanded {
//    unsigned int sequence_number;  /* Incremented with each flow sample
//                                      generated by this source_id.
//                                      Note: If the agent resets the
//                                            sample_pool then it must
//                                            also reset the sequence_number.*/
//    sflow_data_source_expanded source_id; /* sFlowDataSource */
//    unsigned int sampling_rate;    /* sFlowPacketSamplingRate */
//    unsigned int sample_pool;      /* Total number of packets that could have
//                                      been sampled (i.e. packets skipped by
//                                      sampling process + total number of
//                                      samples) */
//    unsigned int drops;            /* Number of times that the sFlow agent
//                                      detected that a packet marked to be
//                                      sampled was dropped due to
//                                      lack of resources. The drops counter
//                                      reports the total number of drops
//                                      detected since the agent was last reset.
//                                      A high drop rate indicates that the
//                                      management agent is unable to process
//                                      samples as fast as they are being
//                                      generated by hardware. Increasing
//                                      sampling_rate will reduce the drop
//                                      rate. Note: An agent that cannot
//                                      detect drops will always report
//                                      zero. */
// 
//    interface_expanded input;      /* Interface packet was received on. */
//    interface_expanded output;     /* Interface packet was sent on. */
// 
//    flow_record flow_records<>;    /* Information about a sampled packet */
// };

public class FlowSampleExpanded implements SampleData {
    public final long sequence_number;
    public final SFlowDataSourceExpanded source_id;
    public final long sampling_rate;
    public final long sample_pool;
    public final long drops;
    public final InterfaceExpanded input;
    public final InterfaceExpanded output;
    public final Array<FlowRecord> flow_records;

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("sequence_number", this.sequence_number)
                .add("source_id", this.source_id)
                .add("sampling_rate", this.sampling_rate)
                .add("sample_pool", this.sample_pool)
                .add("drops", this.drops)
                .add("input", this.input)
                .add("output", this.output)
                .add("flow_records", this.flow_records)
                .toString();
    }

    public FlowSampleExpanded(final ByteBuf buffer) throws InvalidPacketException {
        this.sequence_number = BufferUtils.uint32(buffer);
        this.source_id = new SFlowDataSourceExpanded(buffer);
        this.sampling_rate = BufferUtils.uint32(buffer);
        this.sample_pool = BufferUtils.uint32(buffer);
        this.drops = BufferUtils.uint32(buffer);
        this.input = new InterfaceExpanded(buffer);
        this.output = new InterfaceExpanded(buffer);
        this.flow_records = new Array(buffer, Optional.empty(), FlowRecord::new);
    }

    @Override
    public void writeBson(final BsonWriter bsonWriter, final SampleDatagramEnrichment enr) {
        bsonWriter.writeStartDocument();
        bsonWriter.writeInt64("sequence_number", this.sequence_number);
        bsonWriter.writeName("source_id");
        this.source_id.writeBson(bsonWriter, enr);
        bsonWriter.writeInt64("sampling_rate", this.sampling_rate);
        bsonWriter.writeInt64("sample_pool", this.sample_pool);
        bsonWriter.writeInt64("drops", this.drops);
        bsonWriter.writeName("input");
        this.input.writeBson(bsonWriter, enr);
        bsonWriter.writeName("output");
        this.output.writeBson(bsonWriter, enr);
        bsonWriter.writeStartDocument("flows");
        for (final FlowRecord flowRecord : this.flow_records) {
            bsonWriter.writeName(flowRecord.dataFormat.toId());
            flowRecord.writeBson(bsonWriter, enr);
        }
        bsonWriter.writeEndDocument();
        bsonWriter.writeEndDocument();
    }

    @Override
    public void visit(final SampleDatagramVisitor visitor) {
        visitor.accept(this);
        for (final FlowRecord flowRecord : this.flow_records) {
            flowRecord.visit(visitor);
        }
    }
}
