/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.provision.requisition.command;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.apache.karaf.shell.api.action.Action;
import org.apache.karaf.shell.api.action.Argument;
import org.apache.karaf.shell.api.action.Command;
import org.apache.karaf.shell.api.action.Completion;
import org.apache.karaf.shell.api.action.Option;
import org.apache.karaf.shell.api.action.lifecycle.Reference;
import org.apache.karaf.shell.api.action.lifecycle.Service;
import org.opennms.core.xml.JaxbUtils;
import org.opennms.netmgt.events.api.EventForwarder;
import org.opennms.netmgt.provision.persist.LocationAwareRequisitionClient;
import org.opennms.netmgt.provision.persist.requisition.Requisition;

@Command(scope = "opennms", name = "show-import", description = "Display the resulting requisition generated by a given URL")
@Service
public class ShowImport implements Action {

    @Reference
    private EventForwarder eventForwarder;

    @Option(name = "-l", aliases = "--location", description = "Location", required = false, multiValued = false)
    String location;

    @Option(name = "-s", aliases = "--system-id", description = "System ID")
    String systemId;

    @Option(name = "-t", aliases = "--ttl", description = "Time to live", required = false, multiValued = false)
    Long ttlInMs;

    @Option(name = "-x", aliases = "--xml", description = "XML Output", required = false, multiValued = false)
    boolean xmlOutput = false;

    @Option(name = "-i", aliases = "--import", description = "Import Requisition", required = false, multiValued = false)
    boolean importRequisition = false;

    @Argument(index = 0, name = "type", description = "Type", required = true, multiValued = false)
    @Completion(ProviderTypeNameCompleter.class)
    String type;

    @Argument(index = 1, name = "parameters", description = "Provide parameters in key=value form", multiValued = true)
    List<String> parameters;

    @Reference
    private LocationAwareRequisitionClient client;

    @Override
    public Object execute() throws Exception {
        final CompletableFuture<Requisition> future = client.requisition()
                .withRequisitionProviderType(type)
                .withParameters(parse(parameters))
                .withLocation(location)
                .withSystemId(systemId)
                .withTimeToLive(ttlInMs)
                .execute();

        while (true) {
            try {
                try {
                    Requisition requisition = future.get(1, TimeUnit.SECONDS);
                    if(importRequisition) {
                        System.out.println();
                        ImportRequisition.sendImportRequisitionEvent(eventForwarder, type, parameters, null, false, null);
                        System.out.println();
                    }
                    if (xmlOutput) {
                        return JaxbUtils.marshal(requisition);
                    } else {
                        return requisition;
                    }
                } catch (InterruptedException e) {
                    System.out.println("\nInterrupted.");
                } catch (ExecutionException e) {
                    System.out.printf("\nRequisition retrieval failed with: %s\n", e);
                    e.printStackTrace();
                }
                break;
            } catch (TimeoutException e) {
                // pass
            }
            System.out.print(".");
            System.out.flush();
        }
        return null;
    }

    private static Map<String, String> parse(List<String> attributeList) {
        final Map<String, String> properties = new HashMap<>();
        if (attributeList != null) {
            for (String keyValue : attributeList) {
                int splitAt = keyValue.indexOf("=");
                if (splitAt <= 0) {
                    throw new IllegalArgumentException("Invalid property " + keyValue);
                } else {
                    String key = keyValue.substring(0, splitAt);
                    String value = keyValue.substring(splitAt + 1, keyValue.length());
                    properties.put(key, value);
                }
            }
        }
        return properties;
    }
}
