/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.dao.jaxb;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.opennms.netmgt.config.siteStatusViews.View;
import org.opennms.netmgt.dao.api.SiteStatusViewConfigDao;
import org.springframework.dao.DataRetrievalFailureException;

/**
 * <p>DefaultSiteStatusViewConfigDao class.</p>
 *
 * @author <a href="mailto:brozow@opennms.org">Mathew Brozowski</a>
 * @author <a href="mailto:david@opennms.org">David Hustace</a>
 */
public class DefaultSiteStatusViewConfigDao implements SiteStatusViewConfigDao {
    
    static {
        try {
            SiteStatusViewsFactory.init();
        } catch (FileNotFoundException e) {
            throw new DataRetrievalFailureException("Unable to locate site status view config file", e);
        } catch (IOException e) {
            throw new DataRetrievalFailureException("Error load site status view config file", e);
        }
    }
    
    /**
     * <p>Constructor for DefaultSiteStatusViewConfigDao.</p>
     */
    public DefaultSiteStatusViewConfigDao() {
    }

    /** {@inheritDoc} */
    @Override
    public View getView(String viewName) {
        try {
            return SiteStatusViewsFactory.getInstance().getView(viewName);
        } catch (IOException e) {
            throw new DataRetrievalFailureException("Error load site status view config file", e);
        }
    }

    /**
     * Use this method to get the generated Views class generated by the XSD.
     *
     * @return a {@link org.opennms.netmgt.config.siteStatusViews.Views} object.
     */
    @Override
    public List<View> getViews() {
        return SiteStatusViewsFactory.getConfig().getViews();
    }
    
    /**
     * Use this method to get a Map of view names to marshalled classes based on the generated View class
     * from the XSD.
     *
     * @return <code>Map</> of View classes.
     */
    public Map<String, View> getViewMap() {
        return Collections.unmodifiableMap(SiteStatusViewsFactory.getViewsMap());
    }

    /**
     * <p>getDefaultView</p>
     *
     * @return a {@link org.opennms.netmgt.config.siteStatusViews.View} object.
     */
    @Override
    public View getDefaultView() {
        final String defaultView = SiteStatusViewsFactory.getConfig().getDefaultView();
        return getView(defaultView);
    }

}
