/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.core.ipc.sink.kafka.itests;

import static org.awaitility.Awaitility.await;
import static org.hamcrest.Matchers.greaterThan;
import static org.mockito.Mockito.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.opennms.core.ipc.sink.api.MessageConsumer;
import org.opennms.core.ipc.sink.api.SinkModule;
import org.opennms.core.ipc.sink.api.SyncDispatcher;
import org.opennms.core.ipc.sink.kafka.client.KafkaRemoteMessageDispatcherFactory;
import org.opennms.core.ipc.common.kafka.KafkaSinkConstants;
import org.opennms.core.ipc.sink.kafka.itests.heartbeat.Heartbeat;
import org.opennms.core.ipc.sink.kafka.itests.heartbeat.HeartbeatModule;
import org.opennms.core.ipc.sink.kafka.server.KafkaMessageConsumerManager;
import org.opennms.core.test.OpenNMSJUnit4ClassRunner;
import org.opennms.core.test.kafka.JUnitKafkaServer;
import org.opennms.distributed.core.api.MinionIdentity;
import org.opennms.distributed.core.api.SystemType;
import org.opennms.test.JUnitConfigurationEnvironment;
import org.osgi.service.cm.ConfigurationAdmin;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;

import com.codahale.metrics.ConsoleReporter;
import com.codahale.metrics.Meter;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import com.codahale.metrics.Timer.Context;
import com.google.common.util.concurrent.RateLimiter;

/**
 * Used to help profile the sink producer and consumer
 * against Kafka.
 *
 * By default, we only run a quick test to validate the setup.
 *
 * A longer test, against which you can attach a profiler is available
 * but disabled by default.
 * 
 * @author ranger
 */
@RunWith(OpenNMSJUnit4ClassRunner.class)
@ContextConfiguration(locations={
        "classpath:/META-INF/opennms/applicationContext-soa.xml",
        "classpath:/META-INF/opennms/applicationContext-mockDao.xml",
        "classpath:/META-INF/opennms/applicationContext-proxy-snmp.xml",
        "classpath:/applicationContext-test-ipc-sink-kafka.xml",
        "classpath:/META-INF/opennms/applicationContext-tracer-registry.xml",
        "classpath:/META-INF/opennms/applicationContext-opennms-identity.xml"
})
@JUnitConfigurationEnvironment
public class HeartbeatSinkPerfIT {

    @Rule
    public JUnitKafkaServer kafkaServer = new JUnitKafkaServer();

    @Autowired
    private KafkaMessageConsumerManager consumerManager;

    private KafkaRemoteMessageDispatcherFactory messageDispatcherFactory = new KafkaRemoteMessageDispatcherFactory();

    private List<HeartbeatGenerator> generators = new ArrayList<>();
    private final MetricRegistry metrics = new MetricRegistry();
    private final Meter receivedMeter = metrics.meter("receivedMeter");
    private final Meter sentMeter = metrics.meter("sent");
    private final Timer sendTimer = metrics.timer("send");

    // Tuneables
    private static final int NUM_CONSUMER_THREADS = 2;
    private static final int NUM_GENERATORS = 2;
    private static final double RATE_PER_GENERATOR = 1000.0;

    @Before
    public void setUp() throws Exception {
        Hashtable<String, Object> kafkaConfig = new Hashtable<String, Object>();
        kafkaConfig.put("bootstrap.servers", kafkaServer.getKafkaConnectString());
        ConfigurationAdmin configAdmin = mock(ConfigurationAdmin.class, RETURNS_DEEP_STUBS);
        when(configAdmin.getConfiguration(KafkaSinkConstants.KAFKA_CONFIG_PID).getProperties())
            .thenReturn(kafkaConfig);
        messageDispatcherFactory.setConfigAdmin(configAdmin);
        messageDispatcherFactory.setTracerRegistry(new MockTracerRegistry());
        messageDispatcherFactory.setIdentity(new MinionIdentity() {
            @Override
            public String getId() {
                return "0";
            }
            @Override
            public String getLocation() {
                return "some location";
            }
            @Override
            public String getType() {
                return SystemType.Minion.name();
            }
        });
        messageDispatcherFactory.init();

        System.setProperty(String.format("%sbootstrap.servers", KafkaSinkConstants.KAFKA_CONFIG_SYS_PROP_PREFIX),
                kafkaServer.getKafkaConnectString());
        System.setProperty(String.format("%sauto.offset.reset", KafkaSinkConstants.KAFKA_CONFIG_SYS_PROP_PREFIX),
                "earliest");
        consumerManager.afterPropertiesSet();
    }

    public void configureGenerators() throws Exception {
        System.err.println("Starting Heartbeat generators.");

        // Start the consumer
        final HeartbeatModule parallelHeartbeatModule = new HeartbeatModule() {
            @Override
            public int getNumConsumerThreads() {
                return NUM_CONSUMER_THREADS;
            }
        };
        final HeartbeatConsumer consumer = new HeartbeatConsumer(parallelHeartbeatModule, receivedMeter);
        consumerManager.registerConsumer(consumer);

        // Start the dispatcher
        final SyncDispatcher<Heartbeat> dispatcher = messageDispatcherFactory.createSyncDispatcher(HeartbeatModule.INSTANCE);

        // Fire up the generators
        generators = new ArrayList<>(NUM_GENERATORS);
        for (int k = 0; k < NUM_GENERATORS; k++) {
            final HeartbeatGenerator generator = new HeartbeatGenerator(dispatcher, RATE_PER_GENERATOR, sentMeter, sendTimer);
            generators.add(generator);
            generator.start();
        }
    }

    @After
    public void tearDown() throws Exception {
        if (generators != null) {
            for (HeartbeatGenerator generator : generators) {
                generator.stop();
            }
            generators.clear();
        }
        consumerManager.unregisterAllConsumers();
    }

    @Test(timeout=60000)
    public void quickRun() throws Exception {
        configureGenerators();
        await().atMost(60, TimeUnit.SECONDS).until(() -> Long.valueOf(receivedMeter.getCount()), greaterThan(100L)); 
    }

    @Ignore("enable manually to test long running logging")
    public void longRun() throws Exception {
        // Here we enable console logging of the metrics we gather
        // To see these, you'll want to turn down the logging
        // You can do this by setting the following system property
        // on the JVM when running the tests:
        // -Dorg.opennms.core.test.mockLogger.defaultLogLevel=WARN

        ConsoleReporter reporter = ConsoleReporter.forRegistry(metrics)
                .convertRatesTo(TimeUnit.SECONDS)
                .convertDurationsTo(TimeUnit.MILLISECONDS)
                .build();
        try {
            reporter.start(15, TimeUnit.SECONDS);
            Thread.sleep(5 * 60 * 1000);
        } finally {
            reporter.stop();
        }
    }

    public static class HeartbeatConsumer implements MessageConsumer<Heartbeat,Heartbeat> {

        private final HeartbeatModule heartbeatModule;
        private final Meter receivedMeter;

        public HeartbeatConsumer(HeartbeatModule heartbeatModule, Meter receivedMeter) {
            this.heartbeatModule = heartbeatModule;
            this.receivedMeter = receivedMeter;
        }

        @Override
        public SinkModule<Heartbeat,Heartbeat> getModule() {
            return heartbeatModule;
        }

        @Override
        public void handleMessage(Heartbeat message) {
            receivedMeter.mark();
        }
    }

    public static class HeartbeatGenerator {
        Thread thread;

        final SyncDispatcher<Heartbeat> dispatcher;
        final double rate;
        final AtomicBoolean stopped = new AtomicBoolean(false);
        private final Meter sentMeter;
        private final Timer sendTimer;

        public HeartbeatGenerator(SyncDispatcher<Heartbeat> dispatcher, double rate) {
            this.dispatcher = dispatcher;
            this.rate = rate;
            MetricRegistry metrics = new MetricRegistry();
            this.sentMeter = metrics.meter("sent");
            this.sendTimer = metrics.timer("send");
        }

        public HeartbeatGenerator(SyncDispatcher<Heartbeat> dispatcher, double rate, Meter sentMeter, Timer sendTimer) {
            this.dispatcher = dispatcher;
            this.rate = rate;
            this.sentMeter = sentMeter;
            this.sendTimer = sendTimer;
        }

        public synchronized void start() {
            stopped.set(false);
            final RateLimiter rateLimiter = RateLimiter.create(rate);
            thread = new Thread(new Runnable() {
                @Override
                public void run() {
                   
                    while(!stopped.get()) {
                        rateLimiter.acquire();
                        try (Context ctx = sendTimer.time()) {
                            dispatcher.send(new Heartbeat());
                            sentMeter.mark();
                        }
                    }
                }
            });
            thread.start();
        }

        public synchronized void stop() throws InterruptedException {
            stopped.set(true);
            if (thread != null) {
                thread.join();
                thread = null;
            }
        }
    }
}
