/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.coordinator.group.runtime;

import java.util.List;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.coordinator.group.runtime.CoordinatorEvent;
import org.apache.kafka.coordinator.group.runtime.CoordinatorEventProcessor;
import org.apache.kafka.coordinator.group.runtime.EventAccumulator;
import org.slf4j.Logger;

public class MultiThreadedEventProcessor
implements CoordinatorEventProcessor {
    private final Logger log;
    private final EventAccumulator<TopicPartition, CoordinatorEvent> accumulator;
    private final List<EventProcessorThread> threads;
    private volatile boolean shuttingDown;

    public MultiThreadedEventProcessor(LogContext logContext, String threadPrefix, int numThreads) {
        this.log = logContext.logger(MultiThreadedEventProcessor.class);
        this.shuttingDown = false;
        this.accumulator = new EventAccumulator();
        this.threads = IntStream.range(0, numThreads).mapToObj(threadId -> new EventProcessorThread(threadPrefix + threadId)).collect(Collectors.toList());
        this.threads.forEach(Thread::start);
    }

    @Override
    public void enqueue(CoordinatorEvent event) throws RejectedExecutionException {
        this.accumulator.add(event);
    }

    public synchronized void beginShutdown() {
        if (this.shuttingDown) {
            this.log.debug("Event processor is already shutting down.");
            return;
        }
        this.log.info("Shutting down event processor.");
        this.accumulator.close();
        this.shuttingDown = true;
    }

    @Override
    public void close() throws InterruptedException {
        this.beginShutdown();
        for (Thread thread : this.threads) {
            thread.join();
        }
        this.log.info("Event processor closed.");
    }

    private class EventProcessorThread
    extends Thread {
        private final Logger log;

        EventProcessorThread(String name) {
            super(name);
            this.log = new LogContext("[" + name + "]: ").logger(EventProcessorThread.class);
            this.setDaemon(false);
        }

        private void handleEvents() {
            while (!MultiThreadedEventProcessor.this.shuttingDown) {
                CoordinatorEvent event = (CoordinatorEvent)MultiThreadedEventProcessor.this.accumulator.poll();
                if (event == null) continue;
                try {
                    this.log.debug("Executing event: {}.", (Object)event);
                    event.run();
                }
                catch (Throwable t) {
                    this.log.error("Failed to run event {} due to: {}.", new Object[]{event, t.getMessage(), t});
                    event.complete(t);
                }
                finally {
                    MultiThreadedEventProcessor.this.accumulator.done(event);
                }
            }
        }

        private void drainEvents() {
            CoordinatorEvent event = (CoordinatorEvent)MultiThreadedEventProcessor.this.accumulator.poll(0L, TimeUnit.MILLISECONDS);
            while (event != null) {
                try {
                    this.log.debug("Draining event: {}.", (Object)event);
                    event.complete(new RejectedExecutionException("EventProcessor is closed."));
                }
                catch (Throwable t) {
                    this.log.error("Failed to reject event {} due to: {}.", new Object[]{event, t.getMessage(), t});
                }
                finally {
                    MultiThreadedEventProcessor.this.accumulator.done(event);
                }
                event = (CoordinatorEvent)MultiThreadedEventProcessor.this.accumulator.poll(0L, TimeUnit.MILLISECONDS);
            }
        }

        @Override
        public void run() {
            this.log.info("Starting");
            try {
                this.handleEvents();
            }
            catch (Throwable t) {
                this.log.error("Exiting with exception.", t);
            }
            if (MultiThreadedEventProcessor.this.shuttingDown) {
                this.log.info("Shutting down. Draining the remaining events.");
                try {
                    this.drainEvents();
                }
                catch (Throwable t) {
                    this.log.error("Draining threw exception.", t);
                }
                this.log.info("Shutdown completed");
            }
        }
    }
}

