/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.core.ipc.grpc.server;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import com.codahale.metrics.jmx.JmxReporter;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.ByteString;
import io.grpc.BindableService;
import io.grpc.stub.StreamObserver;
import io.opentracing.Scope;
import io.opentracing.Span;
import io.opentracing.SpanContext;
import io.opentracing.Tracer;
import io.opentracing.propagation.Format;
import io.opentracing.propagation.TextMapAdapter;
import io.opentracing.util.GlobalTracer;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import org.opennms.core.grpc.common.GrpcIpcServer;
import org.opennms.core.ipc.grpc.common.Empty;
import org.opennms.core.ipc.grpc.common.OpenNMSIpcGrpc;
import org.opennms.core.ipc.grpc.common.RpcRequestProto;
import org.opennms.core.ipc.grpc.common.RpcResponseProto;
import org.opennms.core.ipc.grpc.common.SinkMessage;
import org.opennms.core.ipc.sink.api.Message;
import org.opennms.core.ipc.sink.api.MessageConsumerManager;
import org.opennms.core.ipc.sink.api.SinkModule;
import org.opennms.core.ipc.sink.common.AbstractMessageConsumerManager;
import org.opennms.core.logging.Logging;
import org.opennms.core.rpc.api.RemoteExecutionException;
import org.opennms.core.rpc.api.RequestTimedOutException;
import org.opennms.core.rpc.api.RpcClient;
import org.opennms.core.rpc.api.RpcClientFactory;
import org.opennms.core.rpc.api.RpcModule;
import org.opennms.core.rpc.api.RpcRequest;
import org.opennms.core.rpc.api.RpcResponse;
import org.opennms.core.rpc.api.RpcResponseHandler;
import org.opennms.core.tracing.api.TracerRegistry;
import org.opennms.core.tracing.util.TracingInfoCarrier;
import org.opennms.core.utils.PropertiesUtils;
import org.opennms.distributed.core.api.Identity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OpennmsGrpcServer
extends AbstractMessageConsumerManager
implements RpcClientFactory {
    private static final Logger LOG = LoggerFactory.getLogger(OpennmsGrpcServer.class);
    private final GrpcIpcServer grpcIpcServer;
    private String location;
    private Identity identity;
    private long ttl;
    private MetricRegistry rpcMetrics;
    private MetricRegistry sinkMetrics;
    private JmxReporter rpcMetricsReporter;
    private JmxReporter sinkMetricsReporter;
    private TracerRegistry tracerRegistry;
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private final ThreadFactory responseHandlerThreadFactory = new ThreadFactoryBuilder().setNameFormat("rpc-response-handler-%d").build();
    private final ThreadFactory timerThreadFactory = new ThreadFactoryBuilder().setNameFormat("rpc-timeout-tracker-%d").build();
    private final ThreadFactory sinkConsumerThreadFactory = new ThreadFactoryBuilder().setNameFormat("sink-consumer-%d").build();
    private final ExecutorService rpcTimeoutExecutor = Executors.newSingleThreadExecutor(this.timerThreadFactory);
    private final ExecutorService responseHandlerExecutor = Executors.newCachedThreadPool(this.responseHandlerThreadFactory);
    private final Map<String, RpcResponseHandler> rpcResponseMap = new ConcurrentHashMap<String, RpcResponseHandler>();
    private final DelayQueue<RpcResponseHandler> rpcTimeoutQueue = new DelayQueue();
    private final Map<String, StreamObserver<RpcRequestProto>> rpcHandlerByMinionId = new HashMap<String, StreamObserver<RpcRequestProto>>();
    private final Multimap<String, StreamObserver<RpcRequestProto>> rpcHandlerByLocation = LinkedListMultimap.create();
    private final Map<String, Iterator<StreamObserver<RpcRequestProto>>> rpcHandlerIteratorMap = new HashMap<String, Iterator<StreamObserver<RpcRequestProto>>>();
    private final Map<String, SinkModule<?, Message>> sinkModulesById = new ConcurrentHashMap();
    private final Map<String, ExecutorService> sinkConsumersByModuleId = new ConcurrentHashMap<String, ExecutorService>();

    public OpennmsGrpcServer(GrpcIpcServer grpcIpcServer) {
        this.grpcIpcServer = grpcIpcServer;
    }

    public void start() throws IOException {
        try (Logging.MDCCloseable mdc = Logging.withPrefixCloseable((String)"ipc");){
            this.grpcIpcServer.startServer((BindableService)new OpennmsIpcService());
            LOG.info("Added RPC/Sink Service to OpenNMS IPC Grpc Server");
            Properties properties = this.grpcIpcServer.getProperties();
            this.ttl = PropertiesUtils.getProperty((Properties)properties, (String)"ttl", (long)20000L);
            this.rpcTimeoutExecutor.execute(this::handleRpcTimeouts);
            this.rpcMetricsReporter = JmxReporter.forRegistry((MetricRegistry)this.getRpcMetrics()).inDomain("org.opennms.core.ipc.rpc").build();
            this.rpcMetricsReporter.start();
            this.sinkMetricsReporter = JmxReporter.forRegistry((MetricRegistry)this.getRpcMetrics()).inDomain("org.opennms.core.ipc.sink.consumer").build();
            this.sinkMetricsReporter.start();
            if (this.tracerRegistry != null) {
                this.tracerRegistry.init(this.identity.getId());
            }
        }
    }

    protected void startConsumingForModule(SinkModule<?, Message> module) throws Exception {
        if (this.sinkConsumersByModuleId.get(module.getId()) == null) {
            int numOfThreads = OpennmsGrpcServer.getNumConsumerThreads(module);
            ExecutorService executor = Executors.newFixedThreadPool(numOfThreads, this.sinkConsumerThreadFactory);
            this.sinkConsumersByModuleId.put(module.getId(), executor);
            LOG.info("Adding {} consumers for module: {}", (Object)numOfThreads, (Object)module.getId());
        }
        this.sinkModulesById.putIfAbsent(module.getId(), module);
    }

    protected void stopConsumingForModule(SinkModule<?, Message> module) throws Exception {
        ExecutorService executor = this.sinkConsumersByModuleId.get(module.getId());
        if (executor != null) {
            executor.shutdownNow();
        }
        LOG.info("Stopped consumers for module: {}", (Object)module.getId());
        this.sinkModulesById.remove(module.getId());
    }

    public <S extends RpcRequest, T extends RpcResponse> RpcClient<S, T> getClient(final RpcModule<S, T> module) {
        return new RpcClient<S, T>(){

            public CompletableFuture<T> execute(S request) {
                if (request.getLocation() == null || request.getLocation().equals(OpennmsGrpcServer.this.getLocation())) {
                    return module.execute(request);
                }
                Map loggingContext = Logging.getCopyOfContextMap();
                Span span = OpennmsGrpcServer.this.getTracer().buildSpan(module.getId()).start();
                String marshalRequest = module.marshalRequest(request);
                String rpcId = UUID.randomUUID().toString();
                CompletableFuture future = new CompletableFuture();
                Long timeToLive = request.getTimeToLiveMs();
                timeToLive = timeToLive != null && timeToLive > 0L ? timeToLive : OpennmsGrpcServer.this.ttl;
                long expirationTime = System.currentTimeMillis() + timeToLive;
                RpcResponseHandlerImpl responseHandler = new RpcResponseHandlerImpl(future, module, rpcId, request.getLocation(), expirationTime, span, loggingContext);
                OpennmsGrpcServer.this.rpcResponseMap.put(rpcId, responseHandler);
                OpennmsGrpcServer.this.rpcTimeoutQueue.offer(responseHandler);
                RpcRequestProto.Builder builder = RpcRequestProto.newBuilder().setRpcId(rpcId).setLocation(request.getLocation()).setModuleId(module.getId()).setExpirationTime(expirationTime).setRpcContent(ByteString.copyFrom((byte[])marshalRequest.getBytes()));
                if (!Strings.isNullOrEmpty((String)request.getSystemId())) {
                    builder.setSystemId(request.getSystemId());
                }
                this.addTracingInfo((RpcRequest)request, span, builder);
                RpcRequestProto requestProto = builder.build();
                boolean succeeded = OpennmsGrpcServer.this.sendRequest(requestProto);
                this.addMetrics((RpcRequest)request, requestProto.getSerializedSize());
                if (!succeeded) {
                    RpcClientFactory.markFailed((MetricRegistry)OpennmsGrpcServer.this.getRpcMetrics(), (String)request.getLocation(), (String)module.getId());
                    future.completeExceptionally(new RuntimeException("No minion found at location " + request.getLocation()));
                    return future;
                }
                LOG.debug("RPC request from module: {} with RpcId:{} sent to minion at location {}", new Object[]{module.getId(), rpcId, request.getLocation()});
                return future;
            }

            private void addMetrics(RpcRequest request, int messageLen) {
                RpcClientFactory.markRpcCount((MetricRegistry)OpennmsGrpcServer.this.getRpcMetrics(), (String)request.getLocation(), (String)module.getId());
                RpcClientFactory.updateRequestSize((MetricRegistry)OpennmsGrpcServer.this.getRpcMetrics(), (String)request.getLocation(), (String)module.getId(), (int)messageLen);
            }

            private void addTracingInfo(RpcRequest request, Span span, RpcRequestProto.Builder builder) {
                span.setTag("location", request.getLocation());
                if (request.getSystemId() != null) {
                    span.setTag("systemId", request.getSystemId());
                }
                request.getTracingInfo().forEach((arg_0, arg_1) -> ((Span)span).setTag(arg_0, arg_1));
                TracingInfoCarrier tracingInfoCarrier = new TracingInfoCarrier();
                OpennmsGrpcServer.this.getTracer().inject(span.context(), Format.Builtin.TEXT_MAP, (Object)tracingInfoCarrier);
                tracingInfoCarrier.getTracingInfoMap().forEach((arg_0, arg_1) -> ((RpcRequestProto.Builder)builder).putTracingInfo(arg_0, arg_1));
                request.getTracingInfo().forEach((arg_0, arg_1) -> ((RpcRequestProto.Builder)builder).putTracingInfo(arg_0, arg_1));
            }
        };
    }

    private void handleRpcTimeouts() {
        while (!this.closed.get()) {
            try {
                RpcResponseHandler responseHandler = (RpcResponseHandler)this.rpcTimeoutQueue.take();
                if (responseHandler.isProcessed()) continue;
                LOG.warn("RPC request from module: {} with RpcId:{} timedout ", (Object)responseHandler.getRpcModule().getId(), (Object)responseHandler.getRpcId());
                this.responseHandlerExecutor.execute(() -> responseHandler.sendResponse(null));
            }
            catch (InterruptedException e) {
                LOG.info("interrupted while waiting for an element from rpcTimeoutQueue", (Throwable)e);
                Thread.currentThread().interrupt();
                break;
            }
            catch (Exception e) {
                LOG.warn("error while sending response from timeout handler", (Throwable)e);
            }
        }
    }

    private void handleResponse(RpcResponseProto responseProto) {
        if (Strings.isNullOrEmpty((String)responseProto.getRpcId())) {
            return;
        }
        RpcResponseHandler responseHandler = this.rpcResponseMap.get(responseProto.getRpcId());
        if (responseHandler != null) {
            responseHandler.sendResponse(responseProto.getRpcContent().toStringUtf8());
        } else {
            LOG.debug("Received a response for request for module: {} with RpcId:{}, but no outstanding request was found with this id.The request may have timed out", (Object)responseProto.getModuleId(), (Object)responseProto.getRpcId());
        }
    }

    private boolean sendRequest(RpcRequestProto requestProto) {
        StreamObserver<RpcRequestProto> rpcHandler = this.getRpcHandler(requestProto.getLocation(), requestProto.getSystemId());
        if (rpcHandler == null) {
            LOG.warn("No RPC handlers found for location {}", (Object)requestProto.getLocation());
            return false;
        }
        try {
            this.sendRpcRequest(rpcHandler, requestProto);
            return true;
        }
        catch (Throwable e) {
            LOG.error("Encountered exception while sending request {}", (Object)requestProto, (Object)e);
            return false;
        }
    }

    private synchronized void sendRpcRequest(StreamObserver<RpcRequestProto> rpcHandler, RpcRequestProto rpcMessage) {
        rpcHandler.onNext((Object)rpcMessage);
    }

    @VisibleForTesting
    public synchronized StreamObserver<RpcRequestProto> getRpcHandler(String location, String systemId) {
        if (!Strings.isNullOrEmpty((String)systemId)) {
            return this.rpcHandlerByMinionId.get(systemId);
        }
        Iterator<StreamObserver<RpcRequestProto>> iterator = this.rpcHandlerIteratorMap.get(location);
        if (iterator == null) {
            return null;
        }
        return iterator.next();
    }

    private synchronized void addRpcHandler(String location, String systemId, StreamObserver<RpcRequestProto> rpcHandler) {
        if (Strings.isNullOrEmpty((String)location) || Strings.isNullOrEmpty((String)systemId)) {
            LOG.error("Invalid metadata received with location = {} , systemId = {}", (Object)location, (Object)systemId);
            return;
        }
        if (!this.rpcHandlerByLocation.containsValue(rpcHandler)) {
            StreamObserver<RpcRequestProto> obsoleteObserver = this.rpcHandlerByMinionId.get(systemId);
            if (obsoleteObserver != null) {
                this.rpcHandlerByLocation.values().remove(obsoleteObserver);
            }
            this.rpcHandlerByLocation.put((Object)location, rpcHandler);
            this.updateIterator(location);
            this.rpcHandlerByMinionId.put(systemId, rpcHandler);
            LOG.info("Added RPC handler for minion {} at location {}", (Object)systemId, (Object)location);
        }
    }

    private synchronized void updateIterator(String location) {
        Collection streamObservers = this.rpcHandlerByLocation.get((Object)location);
        Iterator iterator = Iterables.cycle((Iterable)streamObservers).iterator();
        this.rpcHandlerIteratorMap.put(location, iterator);
    }

    private synchronized void removeRpcHandler(StreamObserver<RpcRequestProto> rpcHandler) {
        Map.Entry matchingHandler = this.rpcHandlerByLocation.entries().stream().filter(entry -> ((StreamObserver)entry.getValue()).equals(rpcHandler)).findFirst().orElse(null);
        if (matchingHandler != null) {
            this.rpcHandlerByLocation.remove(matchingHandler.getKey(), matchingHandler.getValue());
            this.updateIterator((String)matchingHandler.getKey());
        }
    }

    private boolean isHeaders(RpcResponseProto rpcMessage) {
        return !Strings.isNullOrEmpty((String)rpcMessage.getModuleId()) && rpcMessage.getModuleId().equals("MINION_HEADERS");
    }

    public String getLocation() {
        if (this.location == null && this.getIdentity() != null) {
            return this.getIdentity().getLocation();
        }
        return this.location;
    }

    public void setLocation(String location) {
        this.location = location;
    }

    public Identity getIdentity() {
        return this.identity;
    }

    public void setIdentity(Identity identity) {
        this.identity = identity;
    }

    private MetricRegistry getRpcMetrics() {
        if (this.rpcMetrics == null) {
            this.rpcMetrics = new MetricRegistry();
        }
        return this.rpcMetrics;
    }

    public void setRpcMetrics(MetricRegistry metricRegistry) {
        this.rpcMetrics = metricRegistry;
    }

    public MetricRegistry getSinkMetrics() {
        if (this.sinkMetrics == null) {
            this.sinkMetrics = new MetricRegistry();
        }
        return this.sinkMetrics;
    }

    public void setSinkMetrics(MetricRegistry sinkMetrics) {
        this.sinkMetrics = sinkMetrics;
    }

    public TracerRegistry getTracerRegistry() {
        return this.tracerRegistry;
    }

    public void setTracerRegistry(TracerRegistry tracerRegistry) {
        this.tracerRegistry = tracerRegistry;
    }

    public Tracer getTracer() {
        if (this.tracerRegistry != null) {
            return this.tracerRegistry.getTracer();
        }
        return GlobalTracer.get();
    }

    public void shutdown() {
        this.closed.set(true);
        this.rpcTimeoutQueue.clear();
        this.rpcHandlerByLocation.clear();
        this.rpcHandlerByMinionId.clear();
        this.rpcHandlerIteratorMap.clear();
        this.rpcResponseMap.clear();
        this.sinkModulesById.clear();
        if (this.rpcMetricsReporter != null) {
            this.rpcMetricsReporter.close();
        }
        if (this.sinkMetricsReporter != null) {
            this.sinkMetricsReporter.close();
        }
        this.grpcIpcServer.stopServer();
        this.rpcTimeoutExecutor.shutdownNow();
        this.responseHandlerExecutor.shutdownNow();
        LOG.info("OpenNMS gRPC server stopped");
    }

    @VisibleForTesting
    public Multimap<String, StreamObserver<RpcRequestProto>> getRpcHandlerByLocation() {
        return this.rpcHandlerByLocation;
    }

    private void dispatchSinkMessage(SinkMessage sinkMessage) {
        SinkModule<?, Message> sinkModule = this.sinkModulesById.get(sinkMessage.getModuleId());
        if (sinkModule != null) {
            Message message = sinkModule.unmarshal(sinkMessage.getContent().toByteArray());
            MessageConsumerManager.updateMessageSize((MetricRegistry)this.getSinkMetrics(), (String)sinkMessage.getLocation(), (String)sinkMessage.getModuleId(), (int)sinkMessage.getSerializedSize());
            Timer dispatchTime = MessageConsumerManager.getDispatchTimerMetric((MetricRegistry)this.getSinkMetrics(), (String)sinkMessage.getLocation(), (String)sinkMessage.getModuleId());
            Tracer.SpanBuilder spanBuilder = this.buildSpanFromSinkMessage(sinkMessage);
            try (Scope scope = spanBuilder.startActive(true);
                 Timer.Context context = dispatchTime.time();){
                scope.span().setTag("messageSize", (Number)sinkMessage.getSerializedSize());
                scope.span().setTag("thread", Thread.currentThread().getName());
                this.dispatch(sinkModule, message);
            }
        }
    }

    private Tracer.SpanBuilder buildSpanFromSinkMessage(SinkMessage sinkMessage) {
        Tracer tracer = this.getTracer();
        HashMap tracingInfoMap = new HashMap();
        sinkMessage.getTracingInfoMap().forEach(tracingInfoMap::put);
        SpanContext context = tracer.extract(Format.Builtin.TEXT_MAP, (Object)new TextMapAdapter(tracingInfoMap));
        Tracer.SpanBuilder spanBuilder = context != null ? tracer.buildSpan(sinkMessage.getModuleId()).addReference("follows_from", context) : tracer.buildSpan(sinkMessage.getModuleId());
        return spanBuilder;
    }

    private class OpennmsIpcService
    extends OpenNMSIpcGrpc.OpenNMSIpcImplBase {
        private OpennmsIpcService() {
        }

        public StreamObserver<RpcResponseProto> rpcStreaming(final StreamObserver<RpcRequestProto> responseObserver) {
            return new StreamObserver<RpcResponseProto>(){

                public void onNext(RpcResponseProto rpcResponseProto) {
                    if (OpennmsGrpcServer.this.isHeaders(rpcResponseProto)) {
                        OpennmsGrpcServer.this.addRpcHandler(rpcResponseProto.getLocation(), rpcResponseProto.getSystemId(), (StreamObserver<RpcRequestProto>)responseObserver);
                    } else {
                        OpennmsGrpcServer.this.responseHandlerExecutor.execute(() -> OpennmsGrpcServer.this.handleResponse(rpcResponseProto));
                    }
                }

                public void onError(Throwable throwable) {
                    LOG.error("Error in rpc streaming", throwable);
                }

                public void onCompleted() {
                    LOG.info("Minion RPC handler closed");
                    OpennmsGrpcServer.this.removeRpcHandler((StreamObserver<RpcRequestProto>)responseObserver);
                }
            };
        }

        public StreamObserver<SinkMessage> sinkStreaming(StreamObserver<Empty> responseObserver) {
            return new StreamObserver<SinkMessage>(){

                public void onNext(SinkMessage sinkMessage) {
                    ExecutorService sinkModuleExecutor;
                    if (!Strings.isNullOrEmpty((String)sinkMessage.getModuleId()) && (sinkModuleExecutor = OpennmsGrpcServer.this.sinkConsumersByModuleId.get(sinkMessage.getModuleId())) != null) {
                        sinkModuleExecutor.execute(() -> OpennmsGrpcServer.this.dispatchSinkMessage(sinkMessage));
                    }
                }

                public void onError(Throwable throwable) {
                    LOG.error("Error in sink streaming", throwable);
                }

                public void onCompleted() {
                }
            };
        }
    }

    private class RpcResponseHandlerImpl<S extends RpcRequest, T extends RpcResponse>
    implements RpcResponseHandler {
        private final CompletableFuture<T> responseFuture;
        private final RpcModule<S, T> rpcModule;
        private final String rpcId;
        private final String location;
        private final long expirationTime;
        private final Map<String, String> loggingContext;
        private boolean isProcessed = false;
        private final Long requestCreationTime;
        private final Span span;

        private RpcResponseHandlerImpl(CompletableFuture<T> responseFuture, RpcModule<S, T> rpcModule, String rpcId, String location, long timeout, Span span, Map<String, String> loggingContext) {
            this.responseFuture = responseFuture;
            this.rpcModule = rpcModule;
            this.rpcId = rpcId;
            this.location = location;
            this.expirationTime = timeout;
            this.loggingContext = loggingContext;
            this.span = span;
            this.requestCreationTime = System.currentTimeMillis();
        }

        public void sendResponse(String message) {
            try (Logging.MDCCloseable mdc = Logging.withContextMapCloseable(this.loggingContext);){
                if (message != null) {
                    RpcResponse response = this.rpcModule.unmarshalResponse(message);
                    if (response.getErrorMessage() != null) {
                        this.span.log(response.getErrorMessage());
                        RpcClientFactory.markFailed((MetricRegistry)OpennmsGrpcServer.this.getRpcMetrics(), (String)this.location, (String)this.rpcModule.getId());
                        this.responseFuture.completeExceptionally((Throwable)new RemoteExecutionException(response.getErrorMessage()));
                    } else {
                        this.responseFuture.complete(response);
                    }
                    this.isProcessed = true;
                    RpcClientFactory.updateResponseSize((MetricRegistry)OpennmsGrpcServer.this.getRpcMetrics(), (String)this.location, (String)this.rpcModule.getId(), (int)message.getBytes().length);
                } else {
                    this.span.setTag("timeout", "true");
                    RpcClientFactory.markFailed((MetricRegistry)OpennmsGrpcServer.this.getRpcMetrics(), (String)this.location, (String)this.rpcModule.getId());
                    this.responseFuture.completeExceptionally((Throwable)new RequestTimedOutException((Throwable)new TimeoutException()));
                }
                RpcClientFactory.updateDuration((MetricRegistry)OpennmsGrpcServer.this.getRpcMetrics(), (String)this.location, (String)this.rpcModule.getId(), (long)(System.currentTimeMillis() - this.requestCreationTime));
                OpennmsGrpcServer.this.rpcResponseMap.remove(this.rpcId);
                this.span.finish();
            }
            catch (Throwable e) {
                LOG.error("Error while processing RPC response {}", (Object)message, (Object)e);
            }
            if (this.isProcessed) {
                LOG.debug("RPC Response from module: {} handled successfully for RpcId:{}.", (Object)this.rpcId, (Object)this.rpcModule.getId());
            }
        }

        public boolean isProcessed() {
            return this.isProcessed;
        }

        public String getRpcId() {
            return this.rpcId;
        }

        public int compareTo(Delayed other) {
            long myDelay = this.getDelay(TimeUnit.MILLISECONDS);
            long otherDelay = other.getDelay(TimeUnit.MILLISECONDS);
            return Long.compare(myDelay, otherDelay);
        }

        public long getDelay(TimeUnit unit) {
            long now = System.currentTimeMillis();
            return unit.convert(this.expirationTime - now, TimeUnit.MILLISECONDS);
        }

        public RpcModule<S, T> getRpcModule() {
            return this.rpcModule;
        }
    }
}

