/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.features.kafka.producer;

import java.io.IOException;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.common.serialization.ByteArraySerializer;
import org.opennms.core.ipc.common.kafka.Utils;
import org.opennms.features.kafka.producer.NoOpProducer;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaProducerManager {
    private static final Logger LOG = LoggerFactory.getLogger(KafkaProducerManager.class);
    public static final String GLOBAL_KAFKA_CLIENT_PID = "org.opennms.features.kafka.producer.client";
    public static final String EVENTS_KAFKA_CLIENT_PID = "org.opennms.features.kafka.producer.client.events";
    public static final String ALARMS_KAFKA_CLIENT_PID = "org.opennms.features.kafka.producer.client.alarms";
    public static final String METRICS_KAFKA_CLIENT_PID = "org.opennms.features.kafka.producer.client.metrics";
    public static final String NODES_KAFKA_CLIENT_PID = "org.opennms.features.kafka.producer.client.nodes";
    public static final String TOPOLOGY_KAFKA_CLIENT_PID = "org.opennms.features.kafka.producer.client.topology";
    public static final String ALARM_FEEDBACK_KAFKA_CLIENT_PID = "org.opennms.features.kafka.producer.client.alarmFeedback";
    public static final String BOOTSTRAP_SERVER = "bootstrap.servers";
    private final ConfigurationAdmin configAdmin;
    private final Map<MessageType, Producer<byte[], byte[]>> messageTypeToProducerMap = new ConcurrentHashMap<MessageType, Producer<byte[], byte[]>>();
    private final Map<String, Producer<byte[], byte[]>> pidToProducerMap = new ConcurrentHashMap<String, Producer<byte[], byte[]>>();

    public KafkaProducerManager(ConfigurationAdmin configAdmin) {
        this.configAdmin = Objects.requireNonNull(configAdmin);
    }

    public void init() {
        LOG.info("Initializing KafkaProducerManager");
        for (MessageType messageType : MessageType.values()) {
            try {
                this.getProducerForMessageType(messageType);
                LOG.debug("Successfully initialized producer for message type: {}", (Object)messageType);
            }
            catch (Exception e) {
                LOG.warn("Failed to initialize producer for message type: {}. It will be initialized lazily.", (Object)messageType, (Object)e);
            }
        }
    }

    public void destroy() {
        LOG.info("Destroying KafkaProducerManager");
        this.pidToProducerMap.values().forEach(producer -> {
            try {
                producer.close();
            }
            catch (Exception e) {
                LOG.warn("Error closing Kafka producer", (Throwable)e);
            }
        });
        this.pidToProducerMap.clear();
        this.messageTypeToProducerMap.clear();
    }

    public Producer<byte[], byte[]> getProducerForMessageType(MessageType messageType) {
        return this.messageTypeToProducerMap.computeIfAbsent(messageType, type -> {
            String effectivePid = this.getEffectivePidForMessageType((MessageType)((Object)type));
            if (effectivePid == null) {
                return new NoOpProducer();
            }
            return this.getOrCreateProducerForPid(effectivePid);
        });
    }

    private boolean hasValidConfiguration(String pid) {
        try {
            Configuration config = this.configAdmin.getConfiguration(pid);
            if (config != null && config.getProperties() != null && !config.getProperties().isEmpty()) {
                Dictionary properties = config.getProperties();
                return properties.get(BOOTSTRAP_SERVER) != null;
            }
            return false;
        }
        catch (IOException e) {
            LOG.warn("Failed to check configuration for PID: {}", (Object)pid, (Object)e);
            return false;
        }
    }

    private String getEffectivePidForMessageType(MessageType messageType) {
        String topicSpecificPid = this.getPidForMessageType(messageType);
        if (this.hasValidConfiguration(topicSpecificPid)) {
            LOG.debug("Using topic-specific configuration for {}: {}", (Object)messageType, (Object)topicSpecificPid);
            return topicSpecificPid;
        }
        if (this.hasValidConfiguration(GLOBAL_KAFKA_CLIENT_PID)) {
            LOG.debug("Falling back to global configuration for {}", (Object)messageType);
            return GLOBAL_KAFKA_CLIENT_PID;
        }
        LOG.debug("No configuration available for message type: {}", (Object)messageType);
        return null;
    }

    private String getPidForMessageType(MessageType messageType) {
        switch (messageType) {
            case EVENT: {
                return EVENTS_KAFKA_CLIENT_PID;
            }
            case ALARM: {
                return ALARMS_KAFKA_CLIENT_PID;
            }
            case NODE: {
                return NODES_KAFKA_CLIENT_PID;
            }
            case METRIC: {
                return METRICS_KAFKA_CLIENT_PID;
            }
            case TOPOLOGY_VERTEX: 
            case TOPOLOGY_EDGE: {
                return TOPOLOGY_KAFKA_CLIENT_PID;
            }
            case ALARM_FEEDBACK: {
                return ALARM_FEEDBACK_KAFKA_CLIENT_PID;
            }
        }
        return GLOBAL_KAFKA_CLIENT_PID;
    }

    private String determinePidForMessageType(MessageType messageType) {
        switch (messageType) {
            case EVENT: {
                return this.getEffectivePid(EVENTS_KAFKA_CLIENT_PID);
            }
            case ALARM: {
                return this.getEffectivePid(ALARMS_KAFKA_CLIENT_PID);
            }
            case NODE: {
                return this.getEffectivePid(NODES_KAFKA_CLIENT_PID);
            }
            case METRIC: {
                return this.getEffectivePid(METRICS_KAFKA_CLIENT_PID);
            }
            case TOPOLOGY_VERTEX: 
            case TOPOLOGY_EDGE: {
                return this.getEffectivePid(TOPOLOGY_KAFKA_CLIENT_PID);
            }
            case ALARM_FEEDBACK: {
                return this.getEffectivePid(ALARM_FEEDBACK_KAFKA_CLIENT_PID);
            }
        }
        return GLOBAL_KAFKA_CLIENT_PID;
    }

    private String getEffectivePid(String topicSpecificPid) {
        try {
            Dictionary properties;
            Configuration config = this.configAdmin.getConfiguration(topicSpecificPid);
            if (config != null && config.getProperties() != null && !config.getProperties().isEmpty() && (properties = config.getProperties()).get(BOOTSTRAP_SERVER) != null) {
                LOG.debug("bootstrap.server found  for PID: {}", (Object)topicSpecificPid);
                return topicSpecificPid;
            }
        }
        catch (IOException e) {
            LOG.warn("Failed to check configuration for PID: {}", (Object)topicSpecificPid, (Object)e);
        }
        LOG.debug("Falling back to global configuration for PID: {}", (Object)GLOBAL_KAFKA_CLIENT_PID);
        return GLOBAL_KAFKA_CLIENT_PID;
    }

    private Producer<byte[], byte[]> getOrCreateProducerForPid(String pid) {
        return this.pidToProducerMap.computeIfAbsent(pid, this::initializeProducerForPid);
    }

    private Producer<byte[], byte[]> initializeProducerForPid(String pid) {
        try {
            Properties producerConfig = this.getConfigurationForPid(pid);
            producerConfig.put("key.serializer", ByteArraySerializer.class.getCanonicalName());
            producerConfig.put("value.serializer", ByteArraySerializer.class.getCanonicalName());
            LOG.info("Creating Kafka producer for PID: {} with bootstrap.servers: {}", (Object)pid, (Object)producerConfig.getProperty(BOOTSTRAP_SERVER, "not configured"));
            return (Producer)Utils.runWithGivenClassLoader(() -> new KafkaProducer(producerConfig), (ClassLoader)KafkaProducer.class.getClassLoader());
        }
        catch (Exception e) {
            LOG.error("Failed to create Kafka producer for PID: {}", (Object)pid, (Object)e);
            throw new RuntimeException("Failed to create Kafka producer for PID: " + pid, e);
        }
    }

    public Properties getConfigurationForMessageType(MessageType messageType) {
        String pid = this.determinePidForMessageType(messageType);
        return this.getConfigurationForPid(pid);
    }

    public Properties getConfigurationForPid(String pid) {
        try {
            Properties config = new Properties();
            Dictionary properties = this.configAdmin.getConfiguration(pid).getProperties();
            if (properties != null) {
                Enumeration keys = properties.keys();
                while (keys.hasMoreElements()) {
                    String key = (String)keys.nextElement();
                    Object value = properties.get(key);
                    if (value == null) continue;
                    config.put(key, value);
                }
            }
            return config;
        }
        catch (IOException e) {
            LOG.warn("Failed to load configuration for PID: {}, using empty properties", (Object)pid, (Object)e);
            return new Properties();
        }
    }

    public boolean hasConfigurationForMessageType(MessageType messageType) {
        String effectivePid = this.getEffectivePidForMessageType(messageType);
        return effectivePid != null && this.hasValidConfiguration(effectivePid);
    }

    public ConfigurationAdmin getConfigAdmin() {
        return this.configAdmin;
    }

    public static enum MessageType {
        EVENT,
        ALARM,
        NODE,
        METRIC,
        TOPOLOGY_VERTEX,
        TOPOLOGY_EDGE,
        ALARM_FEEDBACK;

    }
}

