/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.dao.support;

import java.io.File;
import java.io.FileOutputStream;
import java.util.Collections;
import java.util.HashSet;

import org.apache.commons.io.IOUtils;
import org.opennms.netmgt.mock.MockResourceType;
import org.opennms.netmgt.model.OnmsAttribute;
import org.opennms.netmgt.model.OnmsResource;
import org.opennms.netmgt.model.ResourcePath;
import org.opennms.netmgt.model.ResourceTypeUtils;
import org.opennms.netmgt.model.RrdGraphAttribute;
import org.opennms.netmgt.rrd.RrdAttributeType;
import org.opennms.netmgt.rrd.RrdDataSource;
import org.opennms.netmgt.rrd.RrdGraphDetails;
import org.opennms.netmgt.rrd.rrdtool.AbstractJniRrdStrategy;
import org.opennms.netmgt.rrd.rrdtool.MultithreadedJniRrdStrategy;
import org.opennms.test.FileAnticipator;

import junit.framework.TestCase;

/**
 * @author <a href="mailto:dj@opennms.org">DJ Gregor</a>
 */
public class DefaultRrdDaoIntegrationTest extends TestCase {
    private FileAnticipator m_fileAnticipator;

    private AbstractJniRrdStrategy<MultithreadedJniRrdStrategy.CreateCommand, MultithreadedJniRrdStrategy.UpdateCommand> m_rrdStrategy;

    private DefaultRrdDao m_dao;

    @Override
    public void setUp() throws Exception {
        super.setUp();

        m_rrdStrategy = new MultithreadedJniRrdStrategy();
        
        m_fileAnticipator = new FileAnticipator();
        
        m_dao = new DefaultRrdDao();
        m_dao.setRrdStrategy(m_rrdStrategy);
        m_dao.setRrdBaseDirectory(m_fileAnticipator.getTempDir());
        m_dao.setRrdBinaryPath("rrdtool");
        m_dao.afterPropertiesSet();
    }
    
    @Override
    protected void runTest() throws Throwable {
        super.runTest();
        
        if (m_fileAnticipator.isInitialized()) {
            m_fileAnticipator.deleteExpected();
        }
    }

    
    @Override
    protected void tearDown() throws Exception {
        m_fileAnticipator.tearDown();
        
        super.tearDown();
    }

    
    public void testInit() {
        // Don't do anything... test that the setUp method works
    }

    public void testPrintValue() throws Exception {
        long start = System.currentTimeMillis();
        long end = start + (24 * 60 * 60 * 1000);
        
        OnmsResource topResource = new OnmsResource("1", "Node One", new MockResourceType(), new HashSet<OnmsAttribute>(0), new ResourcePath("foo"));

        OnmsAttribute attribute = new RrdGraphAttribute("ifInOctets", "snmp/1/eth0", "ifInOctets.rrd");
        HashSet<OnmsAttribute> attributeSet = new HashSet<OnmsAttribute>(1);
        attributeSet.add(attribute);
        
        MockResourceType childResourceType = new MockResourceType();
        OnmsResource childResource = new OnmsResource("eth0", "Interface One: eth0", childResourceType, attributeSet,  new ResourcePath("foo"));
        childResource.setParent(topResource);
        
        File snmp = m_fileAnticipator.tempDir(ResourceTypeUtils.SNMP_DIRECTORY);
        File node = m_fileAnticipator.tempDir(snmp, topResource.getName());
        File intf = m_fileAnticipator.tempDir(node, childResource.getName());
        
        RrdDataSource rrdDataSource = new RrdDataSource(attribute.getName(), RrdAttributeType.GAUGE, 600, "U", "U");
        MultithreadedJniRrdStrategy.CreateCommand def = m_rrdStrategy.createDefinition("test", intf.getAbsolutePath(), attribute.getName(), 600, Collections.singletonList(rrdDataSource), Collections.singletonList("RRA:AVERAGE:0.5:1:100"));

        m_rrdStrategy.createFile(def);
        File rrdFile = m_fileAnticipator.expecting(intf, attribute.getName() + m_rrdStrategy.getDefaultFileExtension());
        
        MultithreadedJniRrdStrategy.UpdateCommand rrdFileObject = m_rrdStrategy.openFile(rrdFile.getAbsolutePath());
        for (int i = 0; i < 10; i++) {
            m_rrdStrategy.updateFile(rrdFileObject, "test", (start/1000 + 300*i) + ":1");
        }
        m_rrdStrategy.closeFile(rrdFileObject);
        
        Double value = m_dao.getPrintValue(childResource.getAttributes().iterator().next(), "AVERAGE", start, end);
        
        assertNotNull("value should not be null", value);
        assertEquals("value", 1.0, value);
    }
    
    public void testNMS4861() throws Exception
    {
    	//long endTime = 1312775700L;
    	//long endTime = 1312838400L;
    	long endTime = 1312839213L;
    	long startTime = endTime - 86400L;
    	String command = "rrdtool graph -" +
    			" --imgformat PNG" +
    			" --font DEFAULT:7" +
    			" --font TITLE:10" +
    			" --start " + startTime +
    			" --end " + endTime +
    			" --title=\"Netscreen Memory Utilization\"" +
    			" --units-exponent=0 " + 
    			" --lower-limit=0" + 
    			" DEF:value1=netscreen-host-resources.rrd:NetScrnMemAlloc:AVERAGE" +
    			" DEF:value1min=netscreen-host-resources.rrd:NetScrnMemAlloc:MIN" +
    			" DEF:value1max=netscreen-host-resources.rrd:NetScrnMemAlloc:MAX" +
    			" DEF:value2=netscreen-host-resources.rrd:NetScrnMemLeft:AVERAGE" +
    			" DEF:value2min=netscreen-host-resources.rrd:NetScrnMemLeft:MIN" +
    			" DEF:value2max=netscreen-host-resources.rrd:NetScrnMemLeft:MAX" +
    			" DEF:value3=netscreen-host-resources.rrd:NetScrnMemFrag:AVERAGE" +
    			" DEF:value3min=netscreen-host-resources.rrd:NetScrnMemFrag:MIN" +
    			" DEF:value3max=netscreen-host-resources.rrd:NetScrnMemFrag:MAX" +
    			" LINE2:value1#0000ff:\"1  minute\"" + 
    			" GPRINT:value1:AVERAGE:\"Avg \\: %10.2lf\"" + 
    			" GPRINT:value1:MIN:\"Min \\: %10.2lf\"" + 
    			" GPRINT:value1:MAX:\"Max \\: %10.2lf\\n\"" + 
    			" LINE2:value2#00ff00:\"5  minute\"" + 
    			" GPRINT:value2:AVERAGE:\"Avg \\: %10.2lf\"" + 
    			" GPRINT:value2:MIN:\"Min \\: %10.2lf\"" + 
    			" GPRINT:value2:MAX:\"Max \\: %10.2lf\\n\"" + 
    			" LINE2:value3#ff0000:\"15 minute\"" + 
    			" GPRINT:value3:AVERAGE:\"Avg \\: %10.2lf\"" + 
    			" GPRINT:value3:MIN:\"Min \\: %10.2lf\"" + 
    			" GPRINT:value3:MAX:\"Max \\: %10.2lf\\n\"" + 
    			"";
    	
    	final File workDir = new File("src/test/resources");
    	final RrdGraphDetails details = m_rrdStrategy.createGraphReturnDetails(command, workDir);
    	final File outputFile = File.createTempFile("img", "png");
    	IOUtils.copy(details.getInputStream(), new FileOutputStream(outputFile));
    }
}
