/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.poller;

import static org.awaitility.Awaitility.await;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.mock;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.opennms.core.criteria.Criteria;
import org.opennms.core.criteria.CriteriaBuilder;
import org.opennms.core.db.DataSourceFactory;
import org.opennms.core.test.MockLogAppender;
import org.opennms.core.test.OpenNMSJUnit4ClassRunner;
import org.opennms.core.test.db.MockDatabase;
import org.opennms.core.test.db.TemporaryDatabaseAware;
import org.opennms.core.test.db.annotations.JUnitTemporaryDatabase;
import org.opennms.core.utils.Querier;
import org.opennms.netmgt.config.poller.Package;
import org.opennms.netmgt.dao.api.DistPollerDao;
import org.opennms.netmgt.dao.api.MonitoredServiceDao;
import org.opennms.netmgt.dao.api.OutageDao;
import org.opennms.netmgt.dao.mock.MockEventIpcManager;
import org.opennms.netmgt.dao.mock.MockEventIpcManager.SendNowHook;
import org.opennms.netmgt.eventd.AbstractEventUtil;
import org.opennms.netmgt.events.api.EventConstants;
import org.opennms.netmgt.icmp.proxy.LocationAwarePingClient;
import org.opennms.netmgt.mock.MockElement;
import org.opennms.netmgt.mock.MockEventUtil;
import org.opennms.netmgt.mock.MockInterface;
import org.opennms.netmgt.mock.MockNetwork;
import org.opennms.netmgt.mock.MockNode;
import org.opennms.netmgt.mock.MockPersisterFactory;
import org.opennms.netmgt.mock.MockPollerConfig;
import org.opennms.netmgt.mock.MockService;
import org.opennms.netmgt.mock.MockService.SvcMgmtStatus;
import org.opennms.netmgt.mock.MockVisitor;
import org.opennms.netmgt.mock.MockVisitorAdapter;
import org.opennms.netmgt.mock.OutageAnticipator;
import org.opennms.netmgt.mock.PollAnticipator;
import org.opennms.netmgt.model.OnmsMonitoredService;
import org.opennms.netmgt.model.OnmsOutage;
import org.opennms.netmgt.model.events.EventBuilder;
import org.opennms.netmgt.model.events.EventUtils;
import org.opennms.netmgt.poller.pollables.PollableNetwork;
import org.opennms.netmgt.xml.event.Event;
import org.opennms.test.JUnitConfigurationEnvironment;
import org.opennms.test.mock.MockUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.transaction.support.TransactionTemplate;

import com.google.common.collect.Sets;

@RunWith(OpenNMSJUnit4ClassRunner.class)
@ContextConfiguration(locations={
        "classpath:/META-INF/opennms/applicationContext-soa.xml",
        "classpath:/META-INF/opennms/applicationContext-dao.xml",
        "classpath:/META-INF/opennms/applicationContext-mockConfigManager.xml",
        "classpath:/META-INF/opennms/applicationContext-commonConfigs.xml",
        "classpath:/META-INF/opennms/applicationContext-minimal-conf.xml",
        "classpath*:/META-INF/opennms/component-dao.xml",
        "classpath*:/META-INF/opennms/component-service.xml",
        "classpath:/META-INF/opennms/applicationContext-daemon.xml",
        "classpath:/META-INF/opennms/applicationContext-eventUtil.xml",
        "classpath:/META-INF/opennms/mockEventIpcManager.xml",
        "classpath:/META-INF/opennms/applicationContext-pinger.xml",
        "classpath:/META-INF/opennms/applicationContext-rpc-client-mock.xml",
        "classpath:/META-INF/opennms/applicationContext-rpc-poller.xml",

        "classpath:/META-INF/opennms/applicationContext-serviceMonitorRegistry.xml",

        // Override the default QueryManager with the DAO version
        "classpath:/META-INF/opennms/applicationContext-pollerdTest.xml",
        "classpath:/META-INF/opennms/applicationContext-test-deviceConfig.xml"
})
@JUnitConfigurationEnvironment(systemProperties={
        // We don't need a real pinger here
        "org.opennms.netmgt.icmp.pingerClass=org.opennms.netmgt.icmp.NullPinger"
})
@JUnitTemporaryDatabase(tempDbClass=MockDatabase.class,reuseDatabase=false)
public class PollerIT implements TemporaryDatabaseAware<MockDatabase> {

    private Poller m_poller;

    @Autowired
    private MockNetwork m_network;

    private MockDatabase m_db;

    @Autowired
    private MockPollerConfig m_pollerConfig;

    private boolean m_daemonsStarted = false;

    @Autowired
    private MockEventIpcManager m_eventMgr;

    private OutageAnticipator m_outageAnticipator;

    @Autowired
    private QueryManager m_queryManager;

    @Autowired
    private OutageDao m_outageDao;

    @Autowired
    private MonitoredServiceDao m_monitoredServiceDao;

    @Autowired
    private TransactionTemplate m_transactionTemplate;

    @Autowired
    private LocationAwarePollerClient m_locationAwarePollerClient;

    private LocationAwarePingClient m_locationAwarePingClient;

    @Autowired
    DistPollerDao m_distPollerDao;

    //
    // SetUp and TearDown
    //

    @Before
    public void setUp() throws Exception {

        MockUtil.println("------------ Begin Test  --------------------------");
        MockLogAppender.setupLogging();

        m_network.setCriticalService("ICMP");
        m_network.addNode(1, "Router");
        m_network.addInterface("192.168.1.1");
        m_network.addService("ICMP");
        m_network.addService("SMTP");
        m_network.addService("SNMP");
        m_network.addInterface("192.168.1.2");
        m_network.addService("ICMP");
        m_network.addService("SMTP");
        m_network.addNode(2, "Server");
        m_network.addInterface("192.168.1.3");
        m_network.addService("ICMP");
        m_network.addService("HTTP");
        m_network.addService("SMTP");
        m_network.addService("SNMP");
        m_network.addNode(3, "Firewall");
        m_network.addInterface("192.168.1.4");
        m_network.addService("SMTP");
        m_network.addService("HTTP");
        m_network.addInterface("192.168.1.5");
        m_network.addService("SMTP");
        m_network.addService("HTTP");
        m_network.addNode(4, "DownNode");
        m_network.addInterface("192.168.1.6");
        m_network.addService("SNMP");
        m_network.addNode(5, "Loner");
        m_network.addInterface("192.168.1.7");
        m_network.addService("ICMP");
        m_network.addService("SNMP");
        MockService unmonitoredService = m_network.addService("NotMonitored");

        m_db.setDistPoller(m_distPollerDao.whoami().getId());
        m_db.populate(m_network);
        DataSourceFactory.setInstance(m_db);

        m_pollerConfig.setNextOutageIdSql(m_db.getNextOutageIdStatement());
        m_pollerConfig.setNodeOutageProcessingEnabled(true);
        m_pollerConfig.setCriticalService("ICMP");
        m_pollerConfig.addPackage("TestPackage");
        m_pollerConfig.addDowntime(1000L, 0L, -1L, false);
        m_pollerConfig.setDefaultPollInterval(1000L);
        m_pollerConfig.populatePackage(m_network, unmonitoredService);
        m_pollerConfig.addPackage("TestPkg2");
        m_pollerConfig.addDowntime(1000L, 0L, -1L, false);
        m_pollerConfig.setDefaultPollInterval(2000L);
        m_pollerConfig.addService(m_network.getService(2, "192.168.1.3", "HTTP"));

        m_outageAnticipator = new OutageAnticipator(m_db);

        m_eventMgr = new MockEventIpcManager();
        m_eventMgr.setEventWriter(m_db);
        m_eventMgr.setEventAnticipator(m_eventMgr.getEventAnticipator());
        m_eventMgr.addEventListener(m_outageAnticipator);
        m_eventMgr.setSynchronous(false);
        m_eventMgr.setNumSchedulerThreads(2);

        m_locationAwarePingClient = mock(LocationAwarePingClient.class);

        DefaultPollContext pollContext = new DefaultPollContext();
        pollContext.setEventManager(m_eventMgr);
        pollContext.setLocalHostName("localhost");
        pollContext.setName("Test.DefaultPollContext");
        pollContext.setPollerConfig(m_pollerConfig);
        pollContext.setQueryManager(m_queryManager);
        pollContext.setLocationAwarePingClient(m_locationAwarePingClient);

        PollableNetwork network = new PollableNetwork(pollContext);

        m_poller = new Poller();
        m_poller.setMonitoredServiceDao(m_monitoredServiceDao);
        m_poller.setOutageDao(m_outageDao);
        m_poller.setTransactionTemplate(m_transactionTemplate);
        m_poller.setEventIpcManager(m_eventMgr);
        m_poller.setNetwork(network);
        m_poller.setQueryManager(m_queryManager);
        m_poller.setPollerConfig(m_pollerConfig);
        m_poller.setPollOutagesDao(m_pollerConfig);
        m_poller.setLocationAwarePollerClient(m_locationAwarePollerClient);
        m_poller.setServiceMonitorAdaptor((svc, parameters, status) -> status);
        m_poller.setPersisterFactory(new MockPersisterFactory());
    }

    @After
    public void tearDown() throws Exception {
        m_eventMgr.finishProcessingEvents();
        stopDaemons();
        sleep(200);
        m_db.drop();
        MockUtil.println("------------ End Test  --------------------------");
    }

    @Test
    public void testNullInterfaceOnNodeDown() {
        // NODE processing = true;
        m_pollerConfig.setNodeOutageProcessingEnabled(true);
        MockNode node = m_network.getNode(2);
        MockService icmpService = m_network.getService(2, "192.168.1.3", "ICMP");
        MockService smtpService = m_network.getService(2, "192.168.1.3", "SMTP");
        MockService snmpService = m_network.getService(2, "192.168.1.3", "SNMP");

        // start the poller
        startDaemons();

        anticipateDown(node);

        icmpService.bringDown();
        smtpService.bringDown();
        snmpService.bringDown();

        verifyAnticipated(10000);

        // node is down at this point
        boolean foundNodeDown = false;
        for (final Event event : m_eventMgr.getEventAnticipator().getAnticipatedEventsReceived()) {
            if (EventConstants.NODE_DOWN_EVENT_UEI.equals(event.getUei())) {
                foundNodeDown = true;
                assertNull(event.getInterfaceAddress());
            }
        }
        assertTrue(foundNodeDown);
    }

    @Test
    public void testBug709() {

        m_pollerConfig.setNodeOutageProcessingEnabled(true);

        MockNode node = m_network.getNode(2);
        MockService icmpService = m_network.getService(2, "192.168.1.3", "ICMP");
        MockService httpService = m_network.getService(2, "192.168.1.3", "HTTP");

        // start the poller
        startDaemons();

        //
        // Bring Down the HTTP service and expect nodeLostService Event
        //

        resetAnticipated();
        anticipateDown(httpService);

        // bring down the HTTP service
        httpService.bringDown();

        verifyAnticipated(10000);

        //
        // Bring Down the ICMP (on the only interface on the node) now expect
        // nodeDown
        // only.
        //

        resetAnticipated();
        anticipateDown(node);

        // bring down the ICMP service
        icmpService.bringDown();

        // make sure the down events are received
        // verifyAnticipated(10000);
        sleep(5000);
        //
        // Bring up both the node and the httpService at the same time. Expect
        // both a nodeUp and a nodeRegainedService
        //

        resetAnticipated();
        // the order matters here
        anticipateUp(httpService);
        anticipateUp(node);

        // bring up all the services on the node
        node.bringUp();

        // make sure the down events are received
        verifyAnticipated(10000);

    }

    private void resetAnticipated() {
        m_eventMgr.getEventAnticipator().reset();
        m_outageAnticipator.reset();
    }

    @Test
    public void testNodeLostServiceWithReason() {
        m_pollerConfig.setNodeOutageProcessingEnabled(true);

        MockService svc = m_network.getService(1, "192.168.1.1", "ICMP");
        Event e = svc.createDownEvent();
        String reasonParm = "eventReason";
        String val = (String) AbstractEventUtil.getInstance().getNamedParmValue("parm[" + reasonParm + "]", e);
        assertEquals("Service Not Responding.", val);
    }

    @Test
    public void testCritSvcStatusPropagation() {
        m_pollerConfig.setNodeOutageProcessingEnabled(true);

        MockNode node = m_network.getNode(1);

        anticipateDown(node);

        startDaemons();

        bringDownCritSvcs(node);

        verifyAnticipated(8000);
    }

    @Test
    public void testInterfaceWithNoCriticalService() {
        m_pollerConfig.setNodeOutageProcessingEnabled(true);

        MockInterface iface = m_network.getInterface(3, "192.168.1.4");
        MockService svc = iface.getService("SMTP");
        MockService otherService = iface.getService("HTTP");

        startDaemons();

        anticipateDown(iface);

        iface.bringDown();

        verifyAnticipated(8000);

        anticipateUp(iface);
        anticipateDown(otherService, true);

        svc.bringUp();

        verifyAnticipated(8000);

    }

    @Test
    public void testNMS14695() {
        final MockNode testNode = m_network.addNode(999, "Test-Node");
        m_db.writeNode(testNode);
        final MockInterface firstInterface = m_network.addInterface(999, "192.168.42.1");
        m_db.writeInterface(firstInterface);
        final MockService firstIcmp = m_network.addService(999, "192.168.42.1", "ICMP");
        m_db.writeService(firstIcmp);

        m_pollerConfig.setNodeOutageProcessingEnabled(true);
        m_pollerConfig.setCriticalService("ICMP");
        m_pollerConfig.setDefaultPollInterval(1000L);
        m_pollerConfig.addService(firstIcmp);

        startDaemons();

        final MockVisitor gainSvcSender = new MockVisitorAdapter() {
            @Override
            public void visitService(MockService svc) {
                Event event = MockEventUtil.createNodeGainedServiceEvent("Test", svc);
                m_eventMgr.sendEventToListeners(event);
            }
        };

        anticipateDown(testNode);
        firstInterface.bringDown();
        sleep(2000);
        verifyAnticipated(5000);

        final long start = System.currentTimeMillis();
        m_pollerConfig.addScheduledOutage(m_pollerConfig.getPackage("TestPkg2"), "TestOutage", start, start + 60000, firstInterface.getIpAddr());

        m_eventMgr.getEventAnticipator().anticipateEvent(testNode.createUpEvent());

        final MockInterface secondInterface = m_network.addInterface(999, "192.168.84.1");
        m_db.writeInterface(secondInterface);
        final MockService secondIcmp = m_network.addService(999, "192.168.84.1", "ICMP");
        m_db.writeService(secondIcmp);
        m_pollerConfig.addService(secondIcmp);

        testNode.visit(gainSvcSender);

        verifyAnticipated(5000);
    }

    // what about scheduled outages?
    @Test
    public void testDontPollDuringScheduledOutages() {
        long start = System.currentTimeMillis();

        MockInterface iface = m_network.getInterface(1, "192.168.1.2");
        m_pollerConfig.addScheduledOutage(m_pollerConfig.getPackage("TestPackage"), "TestOutage", start, start + 5000, iface.getIpAddr());
        MockUtil.println("Begin Outage");
        startDaemons();

        long now = System.currentTimeMillis();
        sleep(3000 - (now - start));

        MockUtil.println("End Outage");
        assertEquals(0, iface.getPollCount());

        sleep(5000);

        assertTrue(0 < iface.getPollCount());

    }

    // Test harness that tests any type of node, interface or element.
    private void testElementDeleted(MockElement element) {
        Event deleteEvent = element.createDeleteEvent();
        m_pollerConfig.setNodeOutageProcessingEnabled(false);

        PollAnticipator poll = new PollAnticipator();
        element.addAnticipator(poll);

        poll.anticipateAllServices(element);

        startDaemons();

        // wait til after the first poll of the services
        poll.waitForAnticipated(1000L);

        // now delete the node and send a nodeDeleted event
        m_network.removeElement(element);
        m_eventMgr.sendEventToListeners(deleteEvent);

        // reset the poll count and wait to see if any polls on the removed
        // element happened
        m_network.resetInvalidPollCount();

        // now ensure that no invalid polls have occurred
        sleep(3000);

        assertEquals("Received a poll for an element that doesn't exist", 0, m_network.getInvalidPollCount());

    }

    // serviceDeleted: EventConstants.SERVICE_DELETED_EVENT_UEI
    @Test
    public void testServiceDeleted() {
        MockService svc = m_network.getService(1, "192.168.1.1", "SMTP");
        testElementDeleted(svc);
    }

    // interfaceDeleted: EventConstants.INTERFACE_DELETED_EVENT_UEI
    @Test
    public void testInterfaceDeleted() {
        MockInterface iface = m_network.getInterface(1, "192.168.1.1");
        testElementDeleted(iface);
    }

    // nodeDeleted: EventConstants.NODE_DELETED_EVENT_UEI
    @Test
    public void testNodeDeleted() {
        MockNode node = m_network.getNode(1);
        testElementDeleted(node);
    }

    // nodeLabelChanged: EventConstants.NODE_LABEL_CHANGED_EVENT_UEI
    @Test
    public void testNodeLabelChanged() {
        MockNode element = m_network.getNode(1);
        String newLabel = "NEW LABEL";
        Event event = element.createNodeLabelChangedEvent(newLabel);
        m_pollerConfig.setNodeOutageProcessingEnabled(false);

        PollAnticipator poll = new PollAnticipator();
        element.addAnticipator(poll);

        poll.anticipateAllServices(element);

        startDaemons();

        // wait until after the first poll of the services
        poll.waitForAnticipated(1000L);

        assertEquals("Router", m_poller.getNetwork().getNode(1).getNodeLabel());

        element.setLabel(newLabel);
        m_eventMgr.sendEventToListeners(event);

        assertEquals(newLabel, m_poller.getNetwork().getNode(1).getNodeLabel());
    }

    public void testOutagesClosedOnDelete(MockElement element) {

        startDaemons();

        Event deleteEvent = element.createDeleteEvent();

        // bring down so we create an outage in the outages table
        anticipateDown(element);
        element.bringDown();
        verifyAnticipated(5000, false);

        m_outageAnticipator.anticipateOutageClosed(element, deleteEvent);

        // now delete the service
        m_eventMgr.sendEventToListeners(deleteEvent);
        m_network.removeElement(element);

        verifyAnticipated(5000);

    }

    @Test
    public void testServiceOutagesClosedOnDelete() {
        MockService element = m_network.getService(1, "192.168.1.1", "SMTP");
        testOutagesClosedOnDelete(element);

    }

    @Test
    public void testInterfaceOutagesClosedOnDelete() {
        MockInterface element = m_network.getInterface(1, "192.168.1.1");
        testOutagesClosedOnDelete(element);
    }

    @Test
    public void testNodeOutagesClosedOnDelete() {
        MockNode element = m_network.getNode(1);
        testOutagesClosedOnDelete(element);
    }

    // interfaceReparented: EventConstants.INTERFACE_REPARENTED_EVENT_UEI
    @Test
    public void testInterfaceReparented() throws Exception {

        m_pollerConfig.setNodeOutageProcessingEnabled(true);

        MockNode node1 = m_network.getNode(1);
        MockNode node2 = m_network.getNode(2);

        assertNotNull("Node 1 should have 192.168.1.1", node1.getInterface("192.168.1.1"));
        assertNotNull("Node 1 should have 192.168.1.2", node1.getInterface("192.168.1.2"));

        assertNull("Node 2 should not yet have 192.168.1.2", node2.getInterface("192.168.1.2"));
        assertNotNull("Node 2 should have 192.168.1.3", node2.getInterface("192.168.1.3"));

        MockInterface dotTwo = m_network.getInterface(1, "192.168.1.2");
        MockInterface dotThree = m_network.getInterface(2, "192.168.1.3");

        Event reparentEvent = MockEventUtil.createReparentEvent("Test", "192.168.1.2", 1, 2);

        // we are going to reparent to node 2 so when we bring down its only
        // current interface we expect an interface down not the whole node.
        anticipateDown(dotThree);

        startDaemons();

        final int waitTime = 4000;
        final int verifyTime = 4000;

        sleep(waitTime);

        // move the reparented interface and send a reparented event
        dotTwo.moveTo(node2);
        m_db.reparentInterface(dotTwo.getIpAddr(), node1.getNodeId(), node2.getNodeId());

        // send the reparent event to the daemons
        m_eventMgr.sendEventToListeners(reparentEvent);

        sleep(waitTime);

        // now bring down the other interface on the new node
        // System.err.println("Bring Down:"+node2Iface);
        dotThree.bringDown();

        verifyAnticipated(verifyTime);

        resetAnticipated();
        anticipateDown(node2);

        // System.err.println("Bring Down:"+reparentedIface);
        dotTwo.bringDown();

        sleep(waitTime);

        verifyAnticipated(verifyTime);

        node1 = m_network.getNode(1);
        node2 = m_network.getNode(2);

        assertNotNull("Node 1 should still have 192.168.1.1", node1.getInterface("192.168.1.1"));
        assertNull("Node 1 should no longer have 192.168.1.2", node1.getInterface("192.168.1.2"));

        assertNotNull("Node 2 should now have 192.168.1.2", node2.getInterface("192.168.1.2"));
        assertNotNull("Node 2 should still have 192.168.1.3", node2.getInterface("192.168.1.3"));
    }

    // test to see that node lost/regained service events come in
    @Test
    public void testNodeOutageProcessingDisabled() throws Exception {

        m_pollerConfig.setNodeOutageProcessingEnabled(false);

        MockNode node = m_network.getNode(1);

        startDaemons();

        resetAnticipated();
        anticipateServicesDown(node);

        node.bringDown();

        verifyAnticipated(10000);

        resetAnticipated();
        anticipateServicesUp(node);

        node.bringUp();

        verifyAnticipated(10000);

    }

    // test whole node down
    @Test
    public void testNodeOutageProcessingEnabled() throws Exception {

        m_pollerConfig.setNodeOutageProcessingEnabled(true);

        MockNode node = m_network.getNode(1);

        // start the poller
        startDaemons();

        resetAnticipated();
        anticipateDown(node);

        // brind down the node (duh)
        node.bringDown();

        // make sure the correct events are recieved
        verifyAnticipated(10000);

        resetAnticipated();
        anticipateUp(node);

        // bring the node back up
        node.bringUp();

        // make sure the up events are received
        verifyAnticipated(10000);

    }

    @Test
    public void testNodeLostServiceIncludesReason() throws Exception {
        MockService element = m_network.getService(1, "192.168.1.1", "SMTP");
        String expectedReason = "Oh No!! An Outage!!";
        startDaemons();

        resetAnticipated();
        anticipateDown(element);

        MockUtil.println("Bringing down element: " + element);
        element.bringDown(expectedReason);
        MockUtil.println("Finished bringing down element: " + element);

        verifyAnticipated(8000);

        Collection<Event> receivedEvents = m_eventMgr.getEventAnticipator().getAnticipatedEventsReceived();

        assertEquals(2, receivedEvents.size());

        Iterator<Event> receivedEventsIter= receivedEvents.iterator();
        
        Event event1 = receivedEventsIter.next();
        
        assertEquals(expectedReason, EventUtils.getParm(event1, EventConstants.PARM_LOSTSERVICE_REASON));
        
        Event event2 = receivedEventsIter.next();
        
        assertNotNull(event2);
        assertEquals(EventConstants.OUTAGE_CREATED_EVENT_UEI,event2.getUei());
        assertEquals("SMTP",event2.getService());
        assertEquals("192.168.1.1",event2.getInterface());
    }

    @Test
    public void testNodeLostRegainedService() throws Exception {

        testElementDownUp(m_network.getService(1, "192.168.1.1", "SMTP"));

    }

    @Test
    public void testInterfaceDownUp() {

        testElementDownUp(m_network.getInterface(1, "192.168.1.1"));
    }

    @Test
    public void testNodeDownUp() {
        testElementDownUp(m_network.getNode(1));
    }

    private void testElementDownUp(MockElement element) {
        startDaemons();

        resetAnticipated();
        anticipateDown(element);

        MockUtil.println("Bringing down element: " + element);
        element.bringDown();
        MockUtil.println("Finished bringing down element: " + element);

        verifyAnticipated(5000);

        sleep(2000);

        resetAnticipated();
        anticipateUp(element);

        MockUtil.println("Bringing up element: " + element);
        element.bringUp();
        MockUtil.println("Finished bringing up element: " + element);

        verifyAnticipated(8000);
    }

    @Test
    public void testNoEventsOnNoOutages() throws Exception {

        testElementDownUp(m_network.getService(1, "192.168.1.1", "SMTP"));

        resetAnticipated();
        verifyAnticipated(8000, true);

    }

    @Test
    public void testPolling() throws Exception {

        m_pollerConfig.setNodeOutageProcessingEnabled(false);

        // create a poll anticipator
        PollAnticipator anticipator = new PollAnticipator();

        // register it with the interfaces services
        MockInterface iface = m_network.getInterface(1, "192.168.1.2");
        iface.addAnticipator(anticipator);

        //
        // first ensure that polls are working while it is up
        //

        // anticipate three polls on all the interfaces services
        anticipator.anticipateAllServices(iface);
        anticipator.anticipateAllServices(iface);
        anticipator.anticipateAllServices(iface);

        // start the poller
        startDaemons();

        // wait for the polls to occur while its up... 1 poll per second plus overhead
        assertEquals(0, anticipator.waitForAnticipated(4500L).size());


    }

    // test open outages for unmanaged services
    @Test
    public void testUnmangedWithOpenOutageAtStartup() {
        // before we start we need to initialize the database

        // create an outage for the service
        MockService svc = m_network.getService(1, "192.168.1.1", "SMTP");
        MockInterface iface = m_network.getInterface(1, "192.168.1.2");

        Event svcLostEvent = MockEventUtil.createNodeLostServiceEvent("Test", svc);
        m_db.writeEvent(svcLostEvent);
        createOutages(svc, svcLostEvent);

        Event ifaceDownEvent = MockEventUtil.createInterfaceDownEvent("Test", iface);
        m_db.writeEvent(ifaceDownEvent);
        createOutages(iface, ifaceDownEvent);

        // mark the service as unmanaged
        m_db.setServiceStatus(svc, 'U');
        m_db.setInterfaceStatus(iface, 'U');

        // assert that we have an open outage
        assertEquals(1, m_db.countOpenOutagesForService(svc));
        assertEquals(1, m_db.countOutagesForService(svc));

        assertEquals(iface.getServices().size(), m_db.countOutagesForInterface(iface));
        assertEquals(iface.getServices().size(), m_db.countOpenOutagesForInterface(iface));

        startDaemons();

        // assert that we have no open outages
        assertEquals(0, m_db.countOpenOutagesForService(svc));
        assertEquals(1, m_db.countOutagesForService(svc));

        assertEquals(0, m_db.countOpenOutagesForInterface(iface));
        assertEquals(iface.getServices().size(), m_db.countOutagesForInterface(iface));
    }

    @Test
    public void testNodeGainedServiceWhileNodeDownAndServiceUp() {

        startDaemons();

        MockNode node = m_network.getNode(4);
        MockService svc = m_network.getService(4, "192.168.1.6", "SNMP");

        anticipateDown(node);

        node.bringDown();

        verifyAnticipated(5000);

        resetAnticipated();

        anticipateUp(node);
        anticipateDown(svc, true);

        MockService newSvc = m_network.addService(4, "192.168.1.6", "SMTP");

        m_db.writeService(newSvc);

        Event e = MockEventUtil.createNodeGainedServiceEvent("Test", newSvc);
        m_eventMgr.sendEventToListeners(e);

        sleep(5000);
        System.err.println(m_db.getOutages());

        verifyAnticipated(8000);


    }

    @Test
    public void testNodeGainedServiceWhileNodeDownAndServiceDown() {

        startDaemons();

        MockNode node = m_network.getNode(4);
        MockService svc = m_network.getService(4, "192.168.1.6", "SNMP");

        anticipateDown(node);

        node.bringDown();

        verifyAnticipated(5000);

        resetAnticipated();

        MockService newSvc = m_network.addService(4, "192.168.1.6", "SMTP");

        m_db.writeService(newSvc);

        newSvc.bringDown();

        Event e = MockEventUtil.createNodeGainedServiceEvent("Test", newSvc);
        m_eventMgr.sendEventToListeners(e);

        sleep(5000);
        System.err.println(m_db.getOutages());

        verifyAnticipated(8000);

        anticipateUp(node);
        anticipateDown(svc, true);

        newSvc.bringUp();

        verifyAnticipated(5000);


    }

    // test open outages for unmanaged services
    @Test
    public void testReparentCausesStatusChange() {

        m_pollerConfig.setNodeOutageProcessingEnabled(true);

        MockNode node1 = m_network.getNode(1);
        MockNode node2 = m_network.getNode(2);

        MockInterface dotOne = m_network.getInterface(1, "192.168.1.1");
        MockInterface dotTwo = m_network.getInterface(1, "192.168.1.2");
        MockInterface dotThree = m_network.getInterface(2, "192.168.1.3");

        //
        // Plan to bring down both nodes except the reparented interface
        // the node owning the interface should be up while the other is down
        // after reparenting we should got the old owner go down while the other
        // comes up.
        //
        anticipateDown(node2);
        anticipateDown(dotOne);

        // bring down both nodes but bring iface back up
        node1.bringDown();
        node2.bringDown();
        dotTwo.bringUp();

        Event reparentEvent = MockEventUtil.createReparentEvent("Test", "192.168.1.2", 1, 2);

        startDaemons();

        verifyAnticipated(2000);

        m_db.reparentInterface(dotTwo.getIpAddr(), dotTwo.getNodeId(), node2.getNodeId());
        dotTwo.moveTo(node2);

        resetAnticipated();
        anticipateDown(node1, true);
        anticipateUp(node2, true);
        anticipateDown(dotThree, true);

        m_eventMgr.sendEventToListeners(reparentEvent);

        verifyAnticipated(20000);

    }

    // send a nodeGainedService event:
    // EventConstants.NODE_GAINED_SERVICE_EVENT_UEI
    @Test
    public void testSendNodeGainedService() {
        m_pollerConfig.setNodeOutageProcessingEnabled(false);

        startDaemons();

        sendNodeGainedService("SMTP", "HTTP");
    }

    @Test
    public void testSendNodeGainedServiceNodeOutages() {
        m_pollerConfig.setNodeOutageProcessingEnabled(true);

        startDaemons();

        sendNodeGainedService("SMTP", "HTTP");
    }

    @Test
    public void testSendIPv6NodeGainedService() {
        m_pollerConfig.setNodeOutageProcessingEnabled(false);

        startDaemons();

        sendNodeGainedServices(99, "TestNode", "fe80:0000:0000:0000:0231:f982:0123:4567", new String[] { "SMTP", "HTTP" });
    }

    @Test
    public void testSendIPv6NodeGainedServiceNodeOutages() {
        m_pollerConfig.setNodeOutageProcessingEnabled(true);

        startDaemons();
        sendNodeGainedServices(99, "TestNode", "fe80:0000:0000:0000:0231:f982:0123:4567", new String[] { "SMTP", "HTTP" });
    }

    public void sendNodeGainedService(String... svcNames) {
        sendNodeGainedServices(99, "TestNode", "10.1.1.1", svcNames);
    }

    private void sendNodeGainedServices(int nodeid, String nodeLabel, String ipAddr, String... svcNames) {
        assertNotNull(svcNames);
        assertTrue(svcNames.length > 0);

        MockNode node = m_network.addNode(nodeid, nodeLabel);
        m_db.writeNode(node);
        MockInterface iface = m_network.addInterface(nodeid, ipAddr);
        m_db.writeInterface(iface);

        List<MockService> services = new ArrayList<>();
        for(String svcName : svcNames) {
            MockService svc = m_network.addService(nodeid, ipAddr, svcName);
            m_db.writeService(svc);
            m_pollerConfig.addService(svc);
            services.add(svc);
        }

        MockVisitor gainSvcSender = new MockVisitorAdapter() {
            @Override
            public void visitService(MockService svc) {
                Event event = MockEventUtil.createNodeGainedServiceEvent("Test", svc);
                m_eventMgr.sendEventToListeners(event);
            }
        };
        node.visit(gainSvcSender);

        MockService svc1 = services.get(0);

        PollAnticipator anticipator = new PollAnticipator();
        svc1.addAnticipator(anticipator);

        anticipator.anticipateAllServices(svc1);

        final StringBuilder didNotOccur = new StringBuilder();
        for (MockService service : anticipator.waitForAnticipated(10000)) {
            didNotOccur.append(service.toString());
        }
        final StringBuilder unanticipatedStuff = new StringBuilder();
        for (MockService service : anticipator.unanticipatedPolls()) {
            unanticipatedStuff.append(service.toString());
        }

        assertEquals(unanticipatedStuff.toString(), "", didNotOccur.toString());

        anticipateDown(svc1);

        svc1.bringDown();

        verifyAnticipated(10000);
    }

    @Test(timeout=30000)
    public void testSuspendPollingResumeService() {
        MockService svc = m_network.getService(1, "192.168.1.2", "SMTP");
        assertTrue(svc.getPollCount() < 1);

        startDaemons();

        while (svc.getPollCount() < 1) {
            sleep(500);
        }
        assertTrue(0 < svc.getPollCount());

        m_eventMgr.sendEventToListeners(MockEventUtil.createSuspendPollingServiceEvent("Test", svc));
        svc.resetPollCount();

        for (int i = 0; i < 10; i++) {
            // Make sure that the count remains at zero
            assertEquals(0, svc.getPollCount());
            sleep(500);
        }

        m_eventMgr.sendEventToListeners(MockEventUtil.createResumePollingServiceEvent("Test", svc));

        while (svc.getPollCount() < 1) {
            sleep(500);
        }
        assertTrue(0 < svc.getPollCount());

    }

    /**
     * Verifies that outages are properly opened and resolved
     * when events arrive out of order.
     *
     * See NMS-7394 for details.
     */
    @Test
    public void testNoDuplicateOutagesWithDownDownUp() {
        MockInterface nodeIf = m_network.getInterface(1, "192.168.1.1");
        MockService icmpService = m_network.getService(1, "192.168.1.1", "ICMP");
        MockService smtpService = m_network.getService(1, "192.168.1.1", "SMTP");

        // Start the poller
        startDaemons();

        // Kill the critical service on the interface and expect an interfaceDown:
        // The node in question has multiple interfaces, so we don't expect a nodeDown.
        resetAnticipated();
        anticipateDown(nodeIf);

        icmpService.bringDown();

        verifyAnticipated(10000);

        // There should now be a single outage for the SMTP service:
        // The critical service on the interface is down, so all
        // of the services on that interface are also marked as down.
        List<OnmsOutage> smtpOutages = getOutages(smtpService);
        assertEquals(1, smtpOutages.size());
        assertEquals(null, smtpOutages.get(0).getIfRegainedService());

        // Next, we will take the SMTP service offline and bring
        // the ICMP service online in order to make
        // the poller daemon generate a interfaceDown event
        // followed by a nodeLostService event.
        //
        // The poller daemon will then  wait to receive these event back
        // from eventd, so that they are populated with the database ids.
        //
        // When the interfaceDown event is received, it will close
        // the previous outages, and when the nodeLostService event is
        // received it will create a new outage.
        //
        // If these events are received in a different order then which
        // they were sent, we will end up with two outages in the table.
        // This can happen, as observed in NMS-7394, if both events
        // are sent shortly one after another.
        //
        // In order to test the behavior of pollerd, we manually
        // manipulate the order of these events.

        // Stops all other events until the nodeLostService has been processed
        EventOrderAlteringHook hook = new EventOrderAlteringHook(
                EventConstants.NODE_LOST_SERVICE_EVENT_UEI);
        m_eventMgr.setSendNowHook(hook);

        anticipateUp(nodeIf);
        anticipateDown(smtpService);

        smtpService.bringDown();
        icmpService.bringUp();

        verifyAnticipated(10000);

        // There should be two outages in the database:
        // one closed with the event id populated and another one pending
        smtpOutages = getOutages(smtpService);
        assertEquals(2, smtpOutages.size());
        assertNotNull(smtpOutages.get(0).getIfRegainedService());
        assertNotNull(smtpOutages.get(0).getServiceRegainedEvent());
        assertNull(smtpOutages.get(1).getIfRegainedService());
    }

    /**
     * Halts all events until one with the given UEI has
     * been successfully broadcasted.
     *
     * Requires the MockEventIPCManager to asynchronous and to have
     * as many scheduler threads as there are waiting events + 1.
     */
    private static class EventOrderAlteringHook implements SendNowHook {
        private final CountDownLatch m_latch = new CountDownLatch(1);

        private final String m_uei;

        public EventOrderAlteringHook(String uei) {
            m_uei = uei;
        }

        @Override
        public void beforeBroadcast(final Event event) {
            if (m_uei.equalsIgnoreCase(event.getUei())) {
                // pass
            } else {
                try {
                    // stall
                    m_latch.await();
                } catch (InterruptedException e) {
                    // Fail if we're interrupted
                    throw new RuntimeException(e);
                }
            }
        }

        @Override
        public void afterBroadcast(final Event event) {
            if (m_uei.equalsIgnoreCase(event.getUei())) {
                m_latch.countDown();
            }
        }

        @Override
        public void finishProcessingEvents() {
            while(m_latch.getCount() > 0) {
                m_latch.countDown();
            }
        }
    }

    /**
     * Verifies that outages are properly opened and resolved
     * when events arrive out of order.
     *
     * See NMS-7519 for details.
     */
    @Test
    public void testNoDuplicateOutagesWithUpDownDown() throws InterruptedException {
        final MockService httpService = m_network.getService(2, "192.168.1.3", "HTTP");
        final MockService smtpService = m_network.getService(2, "192.168.1.3", "SMTP");

        // Start the poller
        startDaemons();

        // Starting with 0 outages
        List<OnmsOutage> httpOutages = getOutages(httpService);
        assertEquals(0, httpOutages.size());

        // Halts all nodeLostService events until at least 2
        // of them a queued.
        //
        // The event bus should receive the following sequence:
        //   nodeLostService
        //   nodeGainedService
        //   nodeLostService
        //
        // We're looking to reorder them as follows:
        //   nodeGainedService
        //   nodeLostService
        //   nodeLostService
        QueueMultipleDownsHook hook = new QueueMultipleDownsHook(2);
        m_eventMgr.setSendNowHook(hook);

        // Verify that the initial latch count is 2
        waitForHookCount(hook, 2);

        // Take the HTTP service down
        httpService.bringDown();

        // Wait for the latch count to decrease
        waitForHookCount(hook, 1);

        m_eventMgr.getEventAnticipator().reset();
        m_eventMgr.getEventAnticipator().anticipateEvent(httpService.createUpEvent());

        // Bring the HTTP service back up even though the nodeLostService
        // event is still pending
        httpService.bringUp();

        m_eventMgr.getEventAnticipator().waitForAnticipated(10000);

        // Take the HTTP service down again
        httpService.bringDown();

        // Wait for the latch count to decrease and send the queued events
        waitForHookCount(hook, 0);

        m_eventMgr.getEventAnticipator().reset();
        m_eventMgr.getEventAnticipator().anticipateEvent(httpService.createUpEvent());

        // Bring the HTTP service back online
        httpService.bringUp();

        m_eventMgr.getEventAnticipator().waitForAnticipated(10000);

        // We've succeeded in altering the order of events for the
        // HTTP service. Now we make sure that outage processing
        // continues to work as expeceted on a different service
        m_eventMgr.setSendNowHook(null);

        m_eventMgr.getEventAnticipator().reset();
        m_eventMgr.getEventAnticipator().anticipateEvent(httpService.createUpEvent());

        smtpService.bringDown();

        m_eventMgr.getEventAnticipator().waitForAnticipated(10000);

        // Verifies that all of the outage fields are properly
        // set for both outages affecting the HTTP service,
        // even though the events we're send out of order
        httpOutages = getOutages(httpService);
        assertEquals(2, httpOutages.size());
        assertNotNull(httpOutages.get(0).getIfRegainedService());
        assertNotNull(httpOutages.get(0).getIfLostService());
        assertNotNull(httpOutages.get(0).getIfRegainedService());
        assertNotNull(httpOutages.get(0).getServiceRegainedEvent());

        assertNotNull(httpOutages.get(1).getIfRegainedService());
        assertNotNull(httpOutages.get(1).getIfLostService());
        assertNotNull(httpOutages.get(1).getIfRegainedService());
        assertNotNull(httpOutages.get(1).getServiceRegainedEvent());
    }

    /**
     * Halts all "down" events until the desired number
     * of them are queued.
     */
    private static class QueueMultipleDownsHook implements SendNowHook {
        private final CountDownLatch m_latch;

        private final Set<String> m_ueis = Sets.newHashSet();

        public QueueMultipleDownsHook(int count) {
            m_latch = new CountDownLatch(count);
            m_ueis.add(EventConstants.INTERFACE_DOWN_EVENT_UEI);
            m_ueis.add(EventConstants.NODE_DOWN_EVENT_UEI);
            m_ueis.add(EventConstants.NODE_LOST_SERVICE_EVENT_UEI);
        }

        @Override
        public void beforeBroadcast(final Event event) {
            if (!m_ueis.contains(event.getUei())) {
                // pass
            } else {
                try {
                    // stall
                    m_latch.countDown();
                    m_latch.await();
                } catch (InterruptedException e) {
                    // Fail if we're interrupted
                    throw new RuntimeException(e);
                }
            }
        }

        @Override
        public void afterBroadcast(Event event) {
            // pass
        }

        @Override
        public void finishProcessingEvents() {
            while(m_latch.getCount() > 0) {
                m_latch.countDown();
            }
        }

        public long getCount() {
            return m_latch.getCount();
        }
    }

    private void waitForHookCount(QueueMultipleDownsHook hook, long count) throws InterruptedException {
        final long interval = 50;
        long timeout = 10000;
        while(timeout >= 0) {
            Thread.sleep(interval);
            timeout -= interval;
            if (hook.getCount() == count) {
                break;
            }
        }
        assertEquals(count, hook.getCount());
    }

    /**
     * Test for NMS-7426.
     */
    @Test
    public void testServicesWithoutPackagesAreMarkedAsNotPolled() {
        MockService monitoredSvc = m_network.getService(5, "192.168.1.7", "SNMP");
        MockService notMonitoredSvc = m_network.getService(5, "192.168.1.7", "NotMonitored");
        OnmsMonitoredService notMonitored = m_monitoredServiceDao.get(
                notMonitoredSvc.getNodeId(),
                notMonitoredSvc.getAddress(),
                notMonitoredSvc.getSvcName());

        // The status should be set initially set to active
        assertEquals("A", notMonitored.getStatus());

        // Start the poller
        startDaemons();

        // Take a service down, and wait for the event
        // We do this to make ensure the nodes services we're in fact scheduled
        anticipateDown(monitoredSvc);

        monitoredSvc.bringDown();

        verifyAnticipated(10000);

        // The status should now be set to not monitored
        notMonitored = m_monitoredServiceDao.get(
                notMonitoredSvc.getNodeId(),
                notMonitoredSvc.getAddress(),
                notMonitoredSvc.getSvcName());
        assertEquals("N", notMonitored.getStatus());
    }

    @Test
    public void restoresServiceStateOnRestart() {
        MockNode node = m_network.getNode(2);
        MockService criticalService = m_network.getService(2, "192.168.1.3", "ICMP");

        // Bring the critical service down
        anticipateDown(node);
        criticalService.bringDown();

        // Start the poller
        startDaemons();

        // Verify
        verifyAnticipated(10000);

        // Stop the poller
        stopDaemons();

        // Bring the critical service up and expect a nodeUp
        // The service should restore the proper state and know that the outage
        // was triggered by a node down
        anticipateUp(node);
        criticalService.bringUp();

        // (Re)start the poller
        startDaemons();

        // Verify
        verifyAnticipated(10000);
    }

    /**
     * Test for NMS-7585
     */
    @Test
    public void closesOpenOutagesWithNoSvcLostEventIdOnRestart() {
        MockNode node = m_network.getNode(2);
        MockService criticalService = m_network.getService(2, "192.168.1.3", "ICMP");

        // Bring the critical service down
        anticipateDown(node);
        criticalService.bringDown();

        // Start the poller
        startDaemons();

        // Verify
        verifyAnticipated(10000);

        // Stop the poller
        stopDaemons();

        // Remove the reference to the lost service event from all of the outages,
        // and let's pretend that they weren't even there in the first place
        Set<Integer> outageIds = new HashSet<>();
        for (OnmsOutage outage : m_outageDao.findAll()) {
            outage.setServiceLostEvent(null);
            m_outageDao.update(outage);
            outageIds.add(outage.getId());
        }
        m_outageDao.flush();

        // We should get another node down
        m_eventMgr.getEventAnticipator().anticipateEvent(node.createDownEvent());

        // (Re)start the poller
        startDaemons();

        // Verify
        m_eventMgr.getEventAnticipator().waitForAnticipated(10000);

        // Wait for the outages to be populated (this happen after
        // the down event is sent)
        sleep(1000);

        for (OnmsOutage outage : m_outageDao.findAll()) {
            if (outageIds.contains(outage.getId())) {
                // Outages in our list should be closed with
                // no svcRegainedEvent
                assertNotNull(outage.getIfRegainedService());
                assertNull(outage.getServiceRegainedEvent());
            } else {
                // Other outages should be open
                assertNull(outage.getIfRegainedService());
            }
        }
    }

    /**
     * Test for NMS-7761
     */
    @Test
    public void testNoSpuriousNodeDownsOnNodeCategoryMembershipChanged() {
        m_pollerConfig.setNodeOutageProcessingEnabled(true);

        MockNode node = m_network.getNode(1);

        // Start the poller
        startDaemons();

        resetAnticipated();
        anticipateDown(node);

        // Bring down the node (duh)
        node.bringDown();

        // Make sure the correct events are received
        verifyAnticipated(10000);

        // Send a uei.opennms.org/nodes/nodeCategoryMembershipChanged
        EventBuilder eventBuilder = MockEventUtil.createEventBuilder("Test", EventConstants.NODE_CATEGORY_MEMBERSHIP_CHANGED_EVENT_UEI);
        eventBuilder.setNodeid(node.getNodeId());
        Event nodeCatMemChangedEvent = eventBuilder.getEvent();
        m_eventMgr.sendEventToListeners(nodeCatMemChangedEvent);

        // We shouldn't receive any other events
        verifyAnticipated(2000, true);
    }

    /**
     * Test for NMS-9112
     */
    @Test
    public void testNoInvalidPollsAfterNodeCategoryMembershipChanged() throws InterruptedException {
        m_pollerConfig.setNodeOutageProcessingEnabled(true);

        MockNode node = m_network.getNode(1);

        // Start the poller
        startDaemons();

        // Send a uei.opennms.org/nodes/nodeCategoryMembershipChanged
        // This will remove, add or reschedule services for the node as necessary
        EventBuilder eventBuilder = MockEventUtil.createEventBuilder("Test", EventConstants.NODE_CATEGORY_MEMBERSHIP_CHANGED_EVENT_UEI);
        eventBuilder.setNodeid(node.getNodeId());
        Event nodeCatMemChangedEvent = eventBuilder.getEvent();
        m_eventMgr.sendEventToListeners(nodeCatMemChangedEvent);

        // Delete the node and send a nodeDeleted event
        m_network.removeElement(node);
        Event deleteEvent = node.createDeleteEvent();
        m_eventMgr.sendEventToListeners(deleteEvent);

        // Wait a little long than a polling cycle
        sleep(3000);
        m_network.resetInvalidPollCount();

        // Sleep some more and verify that no invalid polls have occurred
        sleep(3000);
        assertEquals("Received a poll for an element that doesn't exist", 0, m_network.getInvalidPollCount());
    }

    @Test
    public void canUpdateLastGoodAndLastFailTimestamps() {
        final Date startOfTest = new Date();

        MockInterface iface = m_network.getInterface(3, "192.168.1.4");
        MockService svc = iface.getService("SMTP");
        assertThat(svc, notNullValue());

        // Start the poller
        startDaemons();

        // Wait until we poll the service
        await().atMost(1, TimeUnit.MINUTES)
                .ignoreExceptions()
                .until(() -> m_monitoredServiceDao.get(svc.getNodeId(), svc.getAddress(), svc.getSvcName()).getLastGood(),
                        greaterThanOrEqualTo(startOfTest));

        // Now take the service offline
        final Date beforeDown = new Date();
        svc.bringDown();

        // Wait until we detect the service to be offline
        await().atMost(1, TimeUnit.MINUTES)
                .ignoreExceptions()
                .until(() -> m_monitoredServiceDao.get(svc.getNodeId(), svc.getAddress(), svc.getSvcName()).getLastFail(),
                        greaterThanOrEqualTo(beforeDown));
    }

    //
    // Utility methods
    //

    private void startDaemons() {
        // m_outageMgr.init();
        m_poller.init();
        // m_outageMgr.start();
        m_poller.start();
        m_daemonsStarted = true;
    }

    private void stopDaemons() {
        if (m_daemonsStarted) {
            m_poller.stop();
            // m_outageMgr.stop();
            m_daemonsStarted = false;
        }
    }

    private void sleep(long millis) {
        try {
            Thread.sleep(millis);
        } catch (final InterruptedException e) {
        }
    }

    private void verifyAnticipated(long millis) {
        verifyAnticipated(millis, true);
    }

    private void verifyAnticipated(long millis, boolean checkUnanticipated) {
        // make sure the down events are received
        MockEventUtil.printEvents("Events we're still waiting for: ", m_eventMgr.getEventAnticipator().waitForAnticipated(millis));
        assertTrue("Expected events not forthcoming", m_eventMgr.getEventAnticipator().waitForAnticipated(0).isEmpty());
        if (checkUnanticipated) {
            sleep(2000);
            MockEventUtil.printEvents("Unanticipated: ", m_eventMgr.getEventAnticipator().getUnanticipatedEvents());
            assertEquals("Received unexpected events", 0, m_eventMgr.getEventAnticipator().getUnanticipatedEvents().size());
        }
        sleep(1000);
        m_eventMgr.finishProcessingEvents();
        assertEquals("Wrong number of outages opened", m_outageAnticipator.getExpectedOpens(), m_outageAnticipator.getActualOpens());
        assertEquals("Wrong number of outages in outage table", m_outageAnticipator.getExpectedOutages(), m_outageAnticipator.getActualOutages());
        assertTrue("Created outages don't match the expected outages", m_outageAnticipator.checkAnticipated());
    }

    private void anticipateUp(MockElement element) {
        anticipateUp(element, false);
    }

    private void anticipateUp(MockElement element, boolean force) {
        if (force || !element.getPollStatus().equals(PollStatus.up())) {
            Event event = element.createUpEvent();
            m_eventMgr.getEventAnticipator().anticipateEvent(event);
            for (Event outageResovedEvent: m_outageAnticipator.anticipateOutageClosed(element, event)) {
                m_eventMgr.getEventAnticipator().anticipateEvent(outageResovedEvent);
            }
        }
    }

    private void anticipateDown(MockElement element) {
        anticipateDown(element, false);
    }

    private void anticipateDown(MockElement element, boolean force) {
        if (force || !element.getPollStatus().equals(PollStatus.down())) {
            Event event = element.createDownEvent();
            m_eventMgr.getEventAnticipator().anticipateEvent(event);
            for (Event outageCretedEvent : m_outageAnticipator.anticipateOutageOpened(element, event)){
                m_eventMgr.getEventAnticipator().anticipateEvent(outageCretedEvent);
            }
        }
    }

    private void anticipateServicesUp(MockElement node) {
        MockVisitor eventCreator = new MockVisitorAdapter() {
            @Override
            public void visitService(MockService svc) {
                anticipateUp(svc);
            }
        };
        node.visit(eventCreator);
    }

    private void anticipateServicesDown(MockElement node) {
        MockVisitor eventCreator = new MockVisitorAdapter() {
            @Override
            public void visitService(MockService svc) {
                anticipateDown(svc);
            }
        };
        node.visit(eventCreator);
    }

    private void createOutages(MockElement element, final Event event) {
        MockVisitor outageCreater = new MockVisitorAdapter() {
            @Override
            public void visitService(MockService svc) {
                if (svc.getMgmtStatus().equals(SvcMgmtStatus.ACTIVE)) {
                    m_db.createOutage(svc, event);
                }
            }
        };
        element.visit(outageCreater);
    }

    private void bringDownCritSvcs(MockElement element) {
        MockVisitor markCritSvcDown = new MockVisitorAdapter() {
            @Override
            public void visitService(MockService svc) {
                if ("ICMP".equals(svc.getSvcName())) {
                    svc.bringDown();
                }
            }
        };
        element.visit(markCritSvcDown);

    }

    /**
     * Retrieves all of the outages (using DAOs) associated with
     * the given service.
     */
    private List<OnmsOutage> getOutages(MockService svc) {
        OnmsMonitoredService monitoredSvc = m_monitoredServiceDao.get(
                svc.getNodeId(), svc.getAddress(), svc.getSvcName());

        Criteria criteria = new CriteriaBuilder(OnmsOutage.class)
            .isNull("perspective")
            .eq("monitoredService", monitoredSvc)
            .orderBy("ifLostService")
            .toCriteria();

        return m_outageDao.findMatching(criteria);
    }

    class OutageChecker extends Querier {
        private Event m_lostSvcEvent;

        private Timestamp m_lostSvcTime;

        private MockService m_svc;

        private Event m_regainedSvcEvent;

        private Timestamp m_regainedSvcTime;

        OutageChecker(MockService svc, Event lostSvcEvent) throws Exception {
            this(svc, lostSvcEvent, null);
        }

        OutageChecker(MockService svc, Event lostSvcEvent,
                      Event regainedSvcEvent) {
            super(m_db, "select * from outages where perspective is null and nodeid = ? and ipAddr = ? and serviceId = ?");

            m_svc = svc;
            m_lostSvcEvent = lostSvcEvent;
            m_lostSvcTime = new Timestamp(m_lostSvcEvent.getTime().getTime());
            m_regainedSvcEvent = regainedSvcEvent;
            if (m_regainedSvcEvent != null) {
                m_regainedSvcTime = new Timestamp(m_regainedSvcEvent.getTime().getTime());
            }
        }

        @Override
        public void processRow(ResultSet rs) throws SQLException {
            assertEquals(m_svc.getNodeId(), rs.getInt("nodeId"));
            assertEquals(m_svc.getIpAddr(), rs.getString("ipAddr"));
            assertEquals(m_svc.getSvcId(), rs.getInt("serviceId"));
            assertEquals(m_lostSvcEvent.getDbid(), Integer.valueOf(rs.getInt("svcLostEventId")));
            assertEquals(m_lostSvcTime, rs.getTimestamp("ifLostService"));
            assertEquals(getRegainedEventId(), rs.getObject("svcRegainedEventId"));
            assertEquals(m_regainedSvcTime, rs.getTimestamp("ifRegainedService"));
        }

        private Long getRegainedEventId() {
            if (m_regainedSvcEvent == null) {
                return null;
            }
            return m_regainedSvcEvent.getDbid();
        }
    }

    @Override
    public void setTemporaryDatabase(MockDatabase database) {
        m_db = database;

    }

    // TODO: test multiple polling packages

    // TODO: test overlapping polling packages

    // TODO: test two packages both with the crit service and status propagation

    // TODO: how does unmanaging a node/iface/service work with the poller

    // TODO: test over lapping poll outages


}
