/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.jasper.grafana;

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;

import javax.imageio.ImageIO;

import org.opennms.core.sysprops.SystemProperties;

import com.google.common.base.Throwables;

/**
 * Utility class to rendering an exception as an image.
 *
 * @author jwhite
 */
public final class ExceptionToPngRenderer {
    private static final String MAX_STACKTRACE_LINES_SYS_PROP = "org.opennms.netmgt.jasper.grafana.maxStackTraceLines";
    private static final int MAX_STACKTRACE_LINES =  SystemProperties.getInteger(MAX_STACKTRACE_LINES_SYS_PROP, 5);

    /**
     * Use the system default font.
     */
    private static final Font font = new Font(null, Font.PLAIN, 48);

    private ExceptionToPngRenderer() {}

    /**
     * Render the given exception to an image.
     *
     * @param e exception
     * @return a byte array of a .png representing the given exception
     * @throws IOException if an error occurs rendering the .png
     */
    public static byte[] renderExceptionToPng(Exception e) {
        // Build a string representation of the exception
        final StringBuilder sb = new StringBuilder();
        sb.append("Exception occurred: ");
        final String stack = Throwables.getStackTraceAsString(e) ;
        // Limit the length of the stack
        sb.append(getFirstNLines(stack, MAX_STACKTRACE_LINES));
        final String text = sb.toString();
        final String[] lines = text.split("\n");

        // Determine the maximum height of any line in the given text
        BufferedImage img = new BufferedImage(1, 1, BufferedImage.TYPE_INT_RGB);
        Graphics2D g2d = img.createGraphics();
        g2d.setFont(font);
        int lineWidth = 0;
        int lineHeight = 0;
        FontMetrics fm = g2d.getFontMetrics();
        for (String line : lines) {
            lineWidth = Math.max(lineWidth, fm.stringWidth(line));
            lineHeight = Math.max(lineHeight, fm.getHeight());
        }
        g2d.dispose();

        // Now render the text to an image line by line
        final int paddingLeft = 10;
        final int paddingTop = 10;
        final int imgWidth = paddingLeft + lineWidth;
        final int imgHeight = paddingTop + lineHeight * lines.length;

        img = new BufferedImage(imgWidth, imgHeight, BufferedImage.TYPE_INT_RGB);
        g2d = img.createGraphics();
        g2d.setFont(font);
        g2d.setColor(Color.WHITE);
        g2d.fillRect(0, 0, imgWidth, imgHeight);
        fm = g2d.getFontMetrics();
        g2d.setColor(Color.BLACK);

        int yOffset = 0;
        for (String line : lines) {
            g2d.drawString(line, paddingLeft, yOffset + fm.getAscent());
            yOffset += lineHeight;
        }
        g2d.dispose();

        // Render the Graphics2D context to a PNG
        final ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            ImageIO.write(img, "png", baos);
        } catch (final IOException ex) {
            // Given that we're writing to a byte array, we don't expect this to ever happen
            throw new IllegalStateException(ex);
        }
        return baos.toByteArray();
    }

    /**
     * Extract the first N lines from the given String.
     */
    private static String getFirstNLines(String s, int numLines) {
        final String[] lines = s.split("\n");
        final StringBuilder sb = new StringBuilder();
        for (int i = 0; i < Math.min(lines.length, numLines); i++) {
            if (i != 0) {
                sb.append("\n");
            }
            sb.append(lines[i]);
        }
        return sb.toString();
    }

}
