/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.bootstrap;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import org.opennms.bootstrap.FilesystemPermissionException;

public class FilesystemPermissionValidator {
    public static boolean VERBOSE = false;
    private static final Random RAND = new Random();
    public static String OPENNMS_HOME = System.getProperty("opennms.home", null);
    public static String DEFAULT_USER = System.getProperty("user.name");
    public static final String[] FULLDIRS = new String[]{"etc", "logs"};
    public static final String[] SPOTDIRS = new String[]{"data", "instances", "share"};
    private static final List<Path> SKIP_PATHS = Arrays.asList(Path.of(".git", new String[0]), Path.of("lost+found", new String[0]));
    private static final FileFilter ONLY_DIRECTORIES = new FileFilter(){

        @Override
        public boolean accept(File pathname) {
            return pathname.isDirectory();
        }
    };

    public boolean validate(String user, Path path) throws FilesystemPermissionException {
        if (path == null) {
            path = Paths.get(OPENNMS_HOME, new String[0]);
        }
        if (VERBOSE) {
            System.out.println("Validating: user " + user + " can write to " + String.valueOf(path) + "\n");
        }
        for (String dir : FULLDIRS) {
            this.validatePath(user, path.resolve(dir));
        }
        for (String dir : SPOTDIRS) {
            Path dirPath = path.resolve(dir);
            if (!dirPath.toFile().exists()) {
                if (!VERBOSE) continue;
                System.out.println("! expected directory " + String.valueOf(dirPath) + " was not found. Skipping.");
                continue;
            }
            if (VERBOSE) {
                System.out.println("- spot-checking " + String.valueOf(dirPath));
            }
            File deepest = null;
            try {
                deepest = dirPath.toRealPath(new LinkOption[0]).toFile();
            }
            catch (IOException e) {
                throw new FilesystemPermissionException(dirPath, user, e);
            }
            while (deepest.isDirectory()) {
                List<File> remainders;
                if (VERBOSE) {
                    System.out.println("  - " + String.valueOf(dirPath) + " is a directory");
                }
                if (SKIP_PATHS.stream().anyMatch(dirPath::endsWith)) {
                    deepest = deepest.getParentFile();
                    continue;
                }
                File[] files = deepest.listFiles(ONLY_DIRECTORIES);
                if (files == null) {
                    if (!VERBOSE) break;
                    System.out.println(String.valueOf(deepest) + " has no contents");
                    break;
                }
                if (files.length == 0) {
                    files = deepest.listFiles();
                }
                if (files.length == 0 || (remainders = Arrays.asList(files).stream().filter(file -> SKIP_PATHS.stream().noneMatch(p -> file.toPath().endsWith((Path)p))).collect(Collectors.toList())).isEmpty()) break;
                deepest = this.getRandomFile(user, deepest, remainders.toArray(new File[0]));
            }
            if (VERBOSE) {
                System.out.println("  - checking " + String.valueOf(deepest));
            }
            this.validateFile(user, deepest.toPath());
        }
        return true;
    }

    private File getRandomFile(String user, File deepest, File[] files) throws FilesystemPermissionException {
        try {
            return files[RAND.nextInt(files.length)].getCanonicalFile();
        }
        catch (IOException e) {
            throw new FilesystemPermissionException(deepest.toPath(), user, e);
        }
    }

    private void validatePath(String user, Path dirPath) throws FilesystemPermissionException {
        if (!dirPath.toFile().exists()) {
            System.err.println("! expected directory " + String.valueOf(dirPath) + " was not found. Skipping.");
            return;
        }
        if (VERBOSE) {
            System.out.println("- checking " + String.valueOf(dirPath));
        }
        FileValidatorVisitor visitor = new FileValidatorVisitor(user);
        try {
            Files.walkFileTree(dirPath, Set.of(FileVisitOption.FOLLOW_LINKS), Integer.MAX_VALUE, visitor);
        }
        catch (IOException e) {
            throw new FilesystemPermissionException(dirPath, user, e);
        }
        visitor.assertValid();
    }

    private void validateFile(String user, Path filePath) throws FilesystemPermissionException {
        File file;
        if (VERBOSE) {
            System.out.println("- validating file " + String.valueOf(filePath) + " is writable by " + user);
        }
        if (!(file = filePath.toFile()).canRead() || !file.canWrite()) {
            throw new FilesystemPermissionException(filePath, user);
        }
        if (file.isDirectory() && !file.canExecute()) {
            throw new FilesystemPermissionException(filePath, user);
        }
    }

    public static void main(String[] args) throws Exception {
        String user = DEFAULT_USER;
        if (args.length >= 1 && args[0] != null) {
            user = args[0];
        }
        Path path = args.length >= 2 && args[1] != null ? Paths.get(args[1], new String[0]) : Paths.get(OPENNMS_HOME, new String[0]);
        if (args.length >= 3 && (Objects.equals(args[2], "-v") || Objects.equals(args[2], "--verbose"))) {
            VERBOSE = true;
        }
        FilesystemPermissionValidator validator = new FilesystemPermissionValidator();
        try {
            validator.validate(user, path);
        }
        catch (FilesystemPermissionException e) {
            System.err.println("\nERROR: at least one file in " + String.valueOf(path) + " is not writable by " + user + ".");
            System.err.println("Make sure $RUNAS in opennms.conf matches the permissions of " + String.valueOf(path) + " and its subdirectories.\n");
            e.printStackTrace();
            System.err.println();
        }
    }

    private final class FileValidatorVisitor
    implements FileVisitor<Path> {
        private final String user;
        private final AtomicReference<FilesystemPermissionException> failure = new AtomicReference();

        public FileValidatorVisitor(String user) {
            this.user = user;
        }

        public void assertValid() throws FilesystemPermissionException {
            if (this.failure.get() == null) {
                return;
            }
            throw this.failure.get();
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            if (SKIP_PATHS.contains(dir.getFileName())) {
                return FileVisitResult.SKIP_SUBTREE;
            }
            try {
                FilesystemPermissionValidator.this.validateFile(this.user, dir);
            }
            catch (FilesystemPermissionException e) {
                this.failure.set(e);
                return FileVisitResult.TERMINATE;
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            try {
                FilesystemPermissionValidator.this.validateFile(this.user, file);
            }
            catch (FilesystemPermissionException e) {
                this.failure.set(e);
                return FileVisitResult.TERMINATE;
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
            if (SKIP_PATHS.contains(file.getFileName())) {
                return FileVisitResult.CONTINUE;
            }
            if (this.failure.get() == null) {
                this.failure.set(new FilesystemPermissionException(file, this.user));
            }
            return FileVisitResult.TERMINATE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            return FileVisitResult.CONTINUE;
        }
    }
}

