/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.core.ipc.twin.common;

import com.codahale.metrics.Counter;
import com.codahale.metrics.MetricRegistry;
import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.fge.jsonpatch.JsonPatch;
import com.github.fge.jsonpatch.JsonPatchException;
import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.InvalidProtocolBufferException;
import io.opentracing.Scope;
import io.opentracing.Span;
import io.opentracing.Tracer;
import io.opentracing.util.GlobalTracer;
import java.io.Closeable;
import java.io.IOException;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import org.opennms.core.ipc.twin.api.TwinRequest;
import org.opennms.core.ipc.twin.api.TwinSubscriber;
import org.opennms.core.ipc.twin.api.TwinUpdate;
import org.opennms.core.ipc.twin.common.AbstractTwinPublisher;
import org.opennms.core.ipc.twin.model.TwinRequestProto;
import org.opennms.core.ipc.twin.model.TwinResponseProto;
import org.opennms.core.tracing.api.TracerRegistry;
import org.opennms.core.tracing.util.TracingInfoCarrier;
import org.opennms.distributed.core.api.Identity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractTwinSubscriber
implements TwinSubscriber {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractTwinSubscriber.class);
    protected static final String TAG_TWIN_RPC_REQUEST = "TwinRpcRequest";
    private static final String TWIN_REQUEST_SENT = "requestSent";
    private static final String TWIN_UPDATE_RECEIVED = "updateReceived";
    private static final String TWIN_UPDATE_DROPPED = "updateDropped";
    private final Identity identity;
    private final Map<String, Subscription> subscriptions = new ConcurrentHashMap<String, Subscription>();
    private final ObjectMapper objectMapper = new ObjectMapper();
    private final TracerRegistry tracerRegistry;
    private final Tracer tracer;
    private final MetricRegistry metrics;
    private final ScheduledExecutorService executorService = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("abstract-twin-subscriber-%d").build());

    protected AbstractTwinSubscriber(Identity identity, TracerRegistry tracerRegistry, MetricRegistry metricRegistry) {
        this.identity = Objects.requireNonNull(identity);
        this.tracerRegistry = tracerRegistry;
        if (tracerRegistry != null) {
            this.tracerRegistry.init(identity.getLocation() + "@" + identity.getId());
            this.tracer = tracerRegistry.getTracer();
        } else {
            this.tracer = GlobalTracer.get();
        }
        this.metrics = metricRegistry != null ? metricRegistry : new MetricRegistry();
    }

    protected abstract void sendRpcRequest(TwinRequest var1);

    public <T> Closeable subscribe(String key, Class<T> clazz, Consumer<T> consumer) {
        if (this.executorService.isShutdown()) {
            throw new IllegalStateException("Subscriber is already closed");
        }
        Subscription subscription = this.subscriptions.computeIfAbsent(key, x$0 -> new Subscription((String)x$0));
        return subscription.consume(clazz, consumer);
    }

    protected void accept(TwinUpdate twinUpdate) {
        if (twinUpdate.getLocation() != null && !twinUpdate.getLocation().equals(this.identity.getLocation())) {
            return;
        }
        if (twinUpdate.getObject() == null || twinUpdate.getSessionId() == null) {
            return;
        }
        String tracingOperationKey = AbstractTwinPublisher.generateTracingOperationKey(twinUpdate.getLocation(), twinUpdate.getKey());
        Tracer.SpanBuilder spanBuilder = TracingInfoCarrier.buildSpanFromTracingMetadata((Tracer)this.getTracer(), (String)tracingOperationKey, (Map)twinUpdate.getTracingInfo(), (String)"follows_from");
        this.executorService.execute(() -> {
            Subscription subscription = this.subscriptions.computeIfAbsent(twinUpdate.getKey(), x$0 -> new Subscription((String)x$0));
            try (Scope scope = spanBuilder.startActive(true);){
                this.addTracingTags(scope.span(), twinUpdate);
                subscription.update(twinUpdate);
            }
            catch (IOException e) {
                LOG.error("Processing update failed: {}", (Object)twinUpdate.getKey(), (Object)e);
                this.updateCounter(MetricRegistry.name((String)twinUpdate.getKey(), (String[])new String[]{TWIN_UPDATE_DROPPED}));
                subscription.request();
            }
        });
    }

    private void addTracingTags(Span span, TwinUpdate twinUpdate) {
        span.setTag("version", (Number)twinUpdate.getVersion());
        span.setTag("sessionId", twinUpdate.getSessionId());
        span.setTag("isPatch", twinUpdate.isPatch());
    }

    private void updateCounter(String counterName) {
        Counter counter = this.metrics.counter(counterName);
        counter.inc();
    }

    protected TwinUpdate mapTwinResponseToProto(byte[] responseBytes) {
        TwinUpdate twinUpdate = new TwinUpdate();
        try {
            TwinResponseProto twinResponseProto = TwinResponseProto.parseFrom(responseBytes);
            if (!Strings.isNullOrEmpty((String)twinResponseProto.getLocation())) {
                twinUpdate.setLocation(twinResponseProto.getLocation());
            }
            if (!Strings.isNullOrEmpty((String)twinResponseProto.getSessionId())) {
                twinUpdate.setSessionId(twinResponseProto.getSessionId());
            }
            twinUpdate.setKey(twinResponseProto.getConsumerKey());
            if (!twinResponseProto.getTwinObject().isEmpty()) {
                twinUpdate.setObject(twinResponseProto.getTwinObject().toByteArray());
            }
            twinUpdate.setPatch(twinResponseProto.getIsPatchObject());
            twinUpdate.setVersion(twinResponseProto.getVersion());
            twinResponseProto.getTracingInfoMap().forEach((arg_0, arg_1) -> ((TwinUpdate)twinUpdate).addTracingInfo(arg_0, arg_1));
            return twinUpdate;
        }
        catch (InvalidProtocolBufferException e) {
            LOG.error("Failed to parse response from proto", (Throwable)e);
            throw new RuntimeException(e);
        }
    }

    protected TwinRequestProto mapTwinRequestToProto(TwinRequest twinRequest) {
        TwinRequestProto.Builder builder = TwinRequestProto.newBuilder();
        builder.setConsumerKey(twinRequest.getKey()).setLocation(this.getIdentity().getLocation()).setSystemId(this.getIdentity().getId());
        twinRequest.getTracingInfo().forEach(builder::putTracingInfo);
        return builder.build();
    }

    public void close() throws IOException {
        this.executorService.shutdown();
        this.subscriptions.clear();
    }

    public Tracer getTracer() {
        return this.tracer;
    }

    public TracerRegistry getTracerRegistry() {
        return this.tracerRegistry;
    }

    public MetricRegistry getMetrics() {
        return this.metrics;
    }

    public Identity getIdentity() {
        return this.identity;
    }

    private class Subscription {
        private final String key;
        private final Set<Consumer<JsonNode>> consumers = Sets.newConcurrentHashSet();
        private Value value = null;
        private ScheduledFuture<?> retry = null;

        private Subscription(String key) {
            this.key = Objects.requireNonNull(key);
        }

        public synchronized <T> Closeable consume(Class<T> clazz, Consumer<T> consumer) {
            Consumer<JsonNode> jsonConsumer = json -> {
                try {
                    Object value = AbstractTwinSubscriber.this.objectMapper.treeToValue((TreeNode)json, clazz);
                    consumer.accept(value);
                }
                catch (Exception e) {
                    LOG.error("Processing twin update failed: {} as {}", new Object[]{this.key, clazz, e});
                }
            };
            if (this.value == null) {
                if (this.retry == null) {
                    this.request();
                }
            } else {
                jsonConsumer.accept(this.value.value);
            }
            this.consumers.add(jsonConsumer);
            return () -> this.consumers.remove(jsonConsumer);
        }

        private synchronized void accept(Value value) {
            Objects.requireNonNull(value);
            if (this.value == null || !Objects.equals(this.value.value, value.value)) {
                this.consumers.forEach(c -> c.accept(value.value));
                AbstractTwinSubscriber.this.updateCounter(MetricRegistry.name((String)this.key, (String[])new String[]{AbstractTwinSubscriber.TWIN_UPDATE_RECEIVED}));
            }
            this.value = value;
        }

        private synchronized void request() {
            String tracingOperationKey = AbstractTwinPublisher.generateTracingOperationKey(AbstractTwinSubscriber.this.getIdentity().getLocation(), this.key);
            Span span = AbstractTwinSubscriber.this.tracer.buildSpan(tracingOperationKey).start();
            TwinRequest request = new TwinRequest(this.key, AbstractTwinSubscriber.this.identity.getLocation());
            this.updateTracingTags(span, request);
            AbstractTwinSubscriber.this.sendRpcRequest(request);
            AbstractTwinSubscriber.this.updateCounter(MetricRegistry.name((String)this.key, (String[])new String[]{AbstractTwinSubscriber.TWIN_REQUEST_SENT}));
            span.finish();
            this.retry = AbstractTwinSubscriber.this.executorService.schedule(this::request, 5L, TimeUnit.SECONDS);
        }

        private void updateTracingTags(Span span, TwinRequest twinRequest) {
            TracingInfoCarrier.updateTracingMetadata((Tracer)AbstractTwinSubscriber.this.getTracer(), (Span)span, (arg_0, arg_1) -> ((TwinRequest)twinRequest).addTracingInfo(arg_0, arg_1));
            span.setTag(AbstractTwinSubscriber.TAG_TWIN_RPC_REQUEST, true);
            span.setTag("location", twinRequest.getLocation());
            span.setTag("systemId", AbstractTwinSubscriber.this.getIdentity().getId());
        }

        public synchronized void update(TwinUpdate update) throws IOException {
            if (this.retry != null) {
                this.retry.cancel(false);
                this.retry = null;
            }
            if (this.value == null || !Objects.equals(this.value.sessionId, update.getSessionId())) {
                if (!update.isPatch()) {
                    this.accept(new Value(update.getSessionId(), update.getVersion(), AbstractTwinSubscriber.this.objectMapper.readTree(update.getObject())));
                } else {
                    AbstractTwinSubscriber.this.updateCounter(MetricRegistry.name((String)this.key, (String[])new String[]{AbstractTwinSubscriber.TWIN_UPDATE_DROPPED}));
                    this.request();
                }
            } else {
                if (update.getVersion() <= this.value.version) {
                    return;
                }
                if (!update.isPatch()) {
                    this.accept(new Value(update.getSessionId(), update.getVersion(), AbstractTwinSubscriber.this.objectMapper.readTree(update.getObject())));
                } else if (update.getVersion() == this.value.version + 1) {
                    try {
                        JsonNode patchObj = AbstractTwinSubscriber.this.objectMapper.readTree(update.getObject());
                        JsonPatch patch = JsonPatch.fromJson((JsonNode)patchObj);
                        JsonNode value = patch.apply(this.value.value);
                        this.accept(new Value(update.getSessionId(), update.getVersion(), value));
                    }
                    catch (JsonPatchException e) {
                        throw new IOException("Unable to apply patch", e);
                    }
                } else {
                    AbstractTwinSubscriber.this.updateCounter(MetricRegistry.name((String)this.key, (String[])new String[]{AbstractTwinSubscriber.TWIN_UPDATE_DROPPED}));
                    this.request();
                }
            }
        }
    }

    private static class Value {
        public final String sessionId;
        public final int version;
        public final JsonNode value;

        private Value(String sessionId, int version, JsonNode value) {
            this.sessionId = Objects.requireNonNull(sessionId);
            this.version = version;
            this.value = Objects.requireNonNull(value);
        }
    }
}

