/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.notifd;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertSame;
import static org.opennms.core.utils.InetAddressUtils.addr;

import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.TreeMap;
import java.util.List;

import org.junit.Assert;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.opennms.netmgt.config.EventConfTestUtil;
import org.opennms.netmgt.config.NotificationManager;
import org.opennms.netmgt.config.api.EventConfDao;
import org.opennms.netmgt.config.notifications.Notification;
import org.opennms.netmgt.events.api.EventConstants;
import org.opennms.netmgt.mock.MockEventUtil;
import org.opennms.netmgt.mock.MockService;
import org.opennms.netmgt.model.EventConfEvent;
import org.opennms.netmgt.model.events.EventBuilder;
import org.opennms.netmgt.xml.event.Event;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.FileSystemResource;

public class BroadcastEventProcessorIT extends NotificationsITCase {

    @Autowired
    private EventConfDao eventConfDao;

    @Before
    @Override
    public void setUp() throws Exception {
        List<EventConfEvent> events = EventConfTestUtil.parseResourcesAsEventConfEvents(
                new FileSystemResource("src/test/resources/org/opennms/netmgt/notifd/eventconf.xml"));
        // Load into DB
        eventConfDao.loadEventsFromDB(events);
        super.setUp();

        m_anticipator.setExpectedDifference(3000);
    }

    @After
    @Override
    public void tearDown() throws Exception {
        super.tearDown(true);
    }

    /**
     * Test calling expandNotifParms to see if the regular expression in
     * m_notifdExpandRE is initialized from {@link BroadcastEventProcessor.NOTIFD_EXPANSION_PARM}.
     */
    @Test
    public void testExpandNotifParms() throws Exception {
        String expandResult = NotificationManager.expandNotifParms("%foo%", new TreeMap<String,String>());
        assertEquals("%foo%", expandResult);

        // This is kinda non-intuitive... but expandNotifParms() only works on whitelisted expansion params
        expandResult = NotificationManager.expandNotifParms("%foo%", Collections.singletonMap("foo", "bar"));
        assertEquals("%foo%", expandResult);

        // The 'noticeid' param is in the whitelist
        expandResult = NotificationManager.expandNotifParms("Notice #%noticeid% RESOLVED: ", Collections.singletonMap("noticeid", "999"));
        assertEquals("Notice #999 RESOLVED: ", expandResult);

        expandResult = NotificationManager.expandNotifParms("RESOLVED: ", Collections.singletonMap("noticeid", "999"));
        assertEquals("RESOLVED: ", expandResult);

        // <notification name="Disk Threshold" status="on"> from bug 2888
        expandResult = NotificationManager.expandNotifParms("Notice %noticeid%: Disk threshold exceeded on %nodelabel%: %parm[all]%.", new TreeMap<String,String>());
        assertEquals("Notice %noticeid%: Disk threshold exceeded on %nodelabel%: %parm[all]%.", expandResult);
        /*
        <event>
            <uei xmlns="">uei.opennms.org/abian/hr-dsk-full</uei>
            <event-label xmlns="">Disk Full</event-label>
            <descr xmlns="">Threshold exceeded for %service% datasource %parm[ds]% on interface %interface%, parms: %parm[all]%</descr>
            <logmsg dest="logndisplay">Threshold exceeded for %service% datasource %parm[ds]% on interface %interface%, parms: %parm[all]%</logmsg>
            <severity xmlns="">Minor</severity>
            <alarm-data reduction-key="%uei%!%nodeid%!%parm[label]%" alarm-type="1" auto-clean="false" />
        </event>
         */
        
        EventBuilder bldr = new EventBuilder(EventConstants.HIGH_THRESHOLD_EVENT_UEI, "testExpandNotifParms");

        bldr.setDescription("High threshold exceeded for %service% datasource %parm[ds]% on interface %interface%, parms: %parm[all]%");
        bldr.setLogMessage("High threshold exceeded for %service% datasource %parm[ds]% on interface %interface%, parms: %parm[all]%");
        bldr.setNodeid(0);
        bldr.setInterface(addr("0.0.0.0"));
        
        bldr.addParam("ds", "dsk-usr-pcent");
        bldr.addParam("value", "Crap! RUN AWAY! We need 20% on the SAN and there's only 15");
        bldr.addParam("threshold", "");
        bldr.addParam("trigger", "");
        bldr.addParam("rearm", "");
        bldr.addParam("label", "");
        bldr.addParam("ifIndex", "");

        /*
        List<String> names = m_notificationManager.getNotificationNames();
        Collections.sort(names);
        for (String name : names) {
            System.out.println(name);
        }
        */
        Notification[] notifications = null;
        notifications = m_notificationManager.getNotifForEvent(null);
        assertNull(notifications);
        notifications = m_notificationManager.getNotifForEvent(bldr.getEvent());
        assertNotNull(notifications);
        assertEquals(1, notifications.length);
        Map<String,String> paramMap = m_eventProcessor.buildParameterMap(notifications[0], bldr.getEvent(), 9999);
        /*
        for (Map.Entry<String,String> entry : paramMap.entrySet()) {
            System.out.println(entry.getKey() + " => " + entry.getValue());
        }
         */
        assertEquals("High disk Threshold exceeded on 0.0.0.0, dsk-usr-pcent with Crap! RUN AWAY! We need 20% on the SAN and there&#39;s only 15%", paramMap.get("-tm"));
        expandResult = NotificationManager.expandNotifParms("Notice #%noticeid%: Disk threshold exceeded on %nodelabel%: %parm[all]%.", paramMap);
        assertEquals("Notice #9999: Disk threshold exceeded on %nodelabel%: %parm[all]%.", expandResult);
    }

    /**
     * Trip a notification and see if the %noticeid% token gets expanded to a numeric
     * value in the subject and text message
     * 
     * @author Jeff Gehlbach <jeffg@jeffg.org>
     */
    @Test
    public void testExpandNoticeId_Bug1745() throws Exception {
        MockService svc = m_network.getService(1, "192.168.1.1", "ICMP");
        Event event = MockEventUtil.createServiceEvent("Test", "uei.opennms.org/test/noticeIdExpansion", svc, null);

        // We need to know what noticeID to expect -- whatever the NotificationManager
        // gives us, the next notice to come out will have noticeID n+1.  This isn't
        // foolproof, but it should work within the confines of JUnit as long as all
        // previous cases have torn down the mock Notifd.
        String antNID = Integer.toString(m_notificationManager.getNoticeId() + 1);

        Date testDate = new Date();
        long finishedNotifs = anticipateNotificationsForGroup("notification '" + antNID + "'", "Notification '" + antNID + "'", "InitialGroup", testDate, 0);
        MockEventUtil.setEventTime(event, testDate);

        m_eventMgr.sendEventToListeners(event);

        verifyAnticipated(finishedNotifs, 1000);
    }

    /**
     * Test Varbindsdecode replacement on notifications.
     * 
     * @author Alejandro Galue <agalue@opennms.org>
     */
    @Test
    public void testVarbindsdecodeNMS7598() throws Exception {
        EventBuilder bldr = new EventBuilder("uei.opennms.org/vendor/Cisco/traps/cHsrpStateChange", "testVarbindsdecodeNMS7598");
        bldr.setNodeid(0);
        bldr.setInterface(addr("0.0.0.0"));
        bldr.addParam("cHsrpGrpStandbyState", "3");
        
        String templateText = "Notice #%noticeid%: new state is %parm[#1]%";
        String expectedText = "Notice #1001: new state is listen(3)";
        Notification n = new Notification();
        n.setName("cHsrpGrpStandbyState");
        n.setSubject(templateText);
        n.setNumericMessage(templateText);
        n.setTextMessage(templateText);
        
        Map<String,String> params = m_eventProcessor.buildParameterMap(n, bldr.getEvent(), 1001);
        Assert.assertEquals(expectedText, params.get(NotificationManager.PARAM_NUM_MSG));
        Assert.assertEquals(expectedText, params.get(NotificationManager.PARAM_TEXT_MSG));
        Assert.assertEquals(expectedText, params.get(NotificationManager.PARAM_SUBJECT));
    }

    @Test
    public void testEventWithoutValidNodeDetailsWhenRuleIsStrict() throws Exception {
        int initialSize = m_notificationDao.countAll();
        EventBuilder builder = new EventBuilder("uei.opennms.org/test/noticeIdExpansion", "test");
        builder.setTime(new Date());
        Event event = builder.getEvent();
        // Don't set node/interface/service for the event.
        m_eventMgr.sendEventToListeners(event);
        // Wait for 5 secs so that event is processed by notifd.
        Thread.sleep(5000);
        int finalSize = m_notificationDao.countAll();
        // This Notification shouldn't get saved.
        assertSame(initialSize, finalSize);
    }
    @Test
    public void testEventWithoutValidNodeDetailsWhenRuleIsNotStrict() throws Exception {
        int initialSize = m_notificationDao.countAll();
        EventBuilder builder = new EventBuilder("uei.opennms.org/test/NoticeWithoutNode", "test");
        builder.setTime(new Date());
        // Don't set node/interface/service for the event.
        Event event = builder.getEvent();
        String antNID = Integer.toString(m_notificationManager.getNoticeId() + 1);
        Date testDate = new Date();
        long finishedNotifs = anticipateNotificationsForGroup("notification '" + antNID + "'", "Notification '" + antNID + "'", "InitialGroup", testDate, 0);
        MockEventUtil.setEventTime(event, testDate);
        m_eventMgr.sendEventToListeners(event);
        // When rule is not strict, when there is no node information to evaluate, it is still considered valid notification
        verifyAnticipated(finishedNotifs, 3000);
    }
}
