/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.config;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertArrayEquals;

import java.io.File;
import java.io.IOException;
import java.util.List;

import org.junit.Test;
import org.opennms.netmgt.config.datacollection.MibObjProperty;
import org.opennms.netmgt.config.datacollection.MibObject;
import org.opennms.test.ThrowableAnticipator;
import org.springframework.core.io.ByteArrayResource;

public class DataCollectionConfigFactoryTest {
	private static final File m_rrdRepository = new File(System.getProperty("java.io.tmpdir") + File.separator + "wonka" + File.separator + "rrd" + File.separator + "snmp");

    private static final String m_xml = "<?xml version=\"1.0\"?>\n" +
            "<datacollection-config\n" +
            "   rrdRepository = \"" + m_rrdRepository.getAbsolutePath() + File.separator + "\">\n" +
            "   <snmp-collection name=\"default\"\n" +
            "       snmpStorageFlag = \"select\">\n" +
            "       <rrd step = \"300\">\n" +
            "           <rra>RRA:AVERAGE:0.5:1:8928</rra>\n" +
            "           <rra>RRA:AVERAGE:0.5:12:8784</rra>\n" +
            "           <rra>RRA:MIN:0.5:12:8784</rra>\n" +
            "           <rra>RRA:MAX:0.5:12:8784</rra>\n" +
            "       </rrd>\n" +
            "       <groups>\n" +
            "           <!-- data from standard (mib-2) sources -->\n" +
            "           <group  name = \"mib2-interfaces\" ifType = \"all\">\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.10\" instance=\"ifIndex\" alias=\"ifInOctets\"    type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.11\" instance=\"ifIndex\" alias=\"ifInUcastpkts\"   type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.12\" instance=\"ifIndex\" alias=\"ifInNUcastpkts\"  type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.13\" instance=\"ifIndex\" alias=\"ifInDiscards\"  type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.14\" instance=\"ifIndex\" alias=\"ifInErrors\"    type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.16\" instance=\"ifIndex\" alias=\"ifOutOctets\"   type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.17\" instance=\"ifIndex\" alias=\"ifOutUcastPkts\"   type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.18\" instance=\"ifIndex\" alias=\"ifOutNUcastPkts\" type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.19\" instance=\"ifIndex\" alias=\"ifOutDiscards\"   type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.20\" instance=\"ifIndex\" alias=\"ifOutErrors\"   type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.31.1.1.1.6\" instance=\"ifIndex\" alias=\"ifHCInOctets\"  type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.31.1.1.1.10\" instance=\"ifIndex\" alias=\"ifHCOutOctets\" type=\"counter\"/>\n" +
            "           </group>\n" +
            "           <group name=\"mib2-tcp\" ifType=\"ignore\">\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.5\" instance=\"0\" alias=\"tcpActiveOpens\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.6\" instance=\"0\" alias=\"tcpPassiveOpens\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.7\" instance=\"0\" alias=\"tcpAttemptFails\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.8\" instance=\"0\" alias=\"tcpEstabResets\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.9\" instance=\"0\" alias=\"tcpCurrEstab\" type=\"Gauge32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.10\" instance=\"0\" alias=\"tcpInSegs\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.11\" instance=\"0\" alias=\"tcpOutSegs\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.12\" instance=\"0\" alias=\"tcpRetransSegs\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.14\" instance=\"0\" alias=\"tcpInErrors\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.15\" instance=\"0\" alias=\"tcpOutRsts\" type=\"Counter32\"/>\n" +
            "           </group>\n" +
            "       </groups>\n" +
            "       <systems>\n" +
            "           <systemDef name = \"Enterprise\">\n" +
            "               <sysoidMask>.1.3.6.1.4.1.</sysoidMask>\n" +
            "               <collect>\n" +
            "                   <includeGroup>mib2-interfaces</includeGroup>\n" +
            "                   <includeGroup>mib2-tcp</includeGroup>\n" +
            "               </collect>\n" +
            "           </systemDef>\n" +
            "\n" +
            "       </systems>\n" +
            "   </snmp-collection>\n" +
            "</datacollection-config>\n" +
            "";

    private static final String m_xml_nms6186 = "<?xml version=\"1.0\"?>\n" +
            "<datacollection-config\n" +
            "   rrdRepository = \"" + m_rrdRepository.getAbsolutePath() + File.separator + "\">\n" +
            "   <snmp-collection name=\"default\"\n" +
            "       snmpStorageFlag = \"select\">\n" +
            "       <rrd step = \"300\">\n" +
            "           <rra>RRA:AVERAGE:0.5:1:8928</rra>\n" +
            "           <rra>RRA:AVERAGE:0.5:12:8784</rra>\n" +
            "           <rra>RRA:MIN:0.5:12:8784</rra>\n" +
            "           <rra>RRA:MAX:0.5:12:8784</rra>\n" +
            "       </rrd>\n" +
            "       <groups>\n" +
            "           <!-- data from standard (mib-2) sources -->\n" +
            "           <group  name = \"mib2-interfaces\" ifType = \"all\">\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.10\" instance=\"ifIndex\" alias=\"ifInOctets\"    type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.11\" instance=\"ifIndex\" alias=\"ifInUcastpkts\"   type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.12\" instance=\"ifIndex\" alias=\"ifInNUcastpkts\"  type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.13\" instance=\"ifIndex\" alias=\"ifInDiscards\"  type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.14\" instance=\"ifIndex\" alias=\"ifInErrors\"    type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.16\" instance=\"ifIndex\" alias=\"ifOutOctets\"   type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.17\" instance=\"ifIndex\" alias=\"ifOutUcastPkts\"   type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.18\" instance=\"ifIndex\" alias=\"ifOutNUcastPkts\" type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.19\" instance=\"ifIndex\" alias=\"ifOutDiscards\"   type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.2.2.1.20\" instance=\"ifIndex\" alias=\"ifOutErrors\"   type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.31.1.1.1.6\" instance=\"ifIndex\" alias=\"ifHCInOctets\"  type=\"counter\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.31.1.1.1.10\" instance=\"ifIndex\" alias=\"ifHCOutOctets\" type=\"counter\"/>\n" +
            "           </group>\n" +
            "           <group name=\"mib2-tcp\" ifType=\"ignore\">\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.5\" instance=\"0\" alias=\"tcpActiveOpens\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.6\" instance=\"0\" alias=\"tcpPassiveOpens\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.7\" instance=\"0\" alias=\"tcpAttemptFails\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.8\" instance=\"0\" alias=\"tcpEstabResets\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.9\" instance=\"0\" alias=\"tcpCurrEstab\" type=\"Gauge32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.10\" instance=\"0\" alias=\"tcpInSegs\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.11\" instance=\"0\" alias=\"tcpOutSegs\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.12\" instance=\"0\" alias=\"tcpRetransSegs\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.14\" instance=\"0\" alias=\"tcpInErrors\" type=\"Counter32\"/>\n" +
            "             <mibObj oid=\".1.3.6.1.2.1.6.15\" instance=\"0\" alias=\"tcpOutRsts\" type=\"Counter32\"/>\n" +
            "           </group>\n" +
            "       </groups>\n" +
            "   </snmp-collection>\n" +
            "</datacollection-config>\n" +
            "";

    private static final String m_systemsXmlFragment =
            "       <systems>\n" +
            "           <systemDef name = \"Enterprise\">\n" +
            "               <sysoidMask>.1.3.6.1.4.1.</sysoidMask>\n" +
            "               <collect>\n" +
            "                   <includeGroup>mib2-interfaces</includeGroup>\n" +
            "                   <includeGroup>mib2-tcp</includeGroup>\n" +
            "               </collect>\n" +
            "           </systemDef>\n" +
            "       </systems>\n";

    private static final String m_brocadeXmlFragment =
    "       <resourceType name=\"brocadeIndex\" label=\"Brocade Switches\">\n" +
    "         <persistenceSelectorStrategy class=\"foo\"/>\n" +
    "         <storageStrategy class=\"foo\"/>\n" +
    "       </resourceType>\n";

    @Test
    public void testSetInstance() throws IOException {
        initDataCollectionFactory(m_xml);
        assertEquals(m_rrdRepository.getAbsolutePath(), DataCollectionConfigFactory.getInstance().getRrdPath());
        assertEquals(0, DataCollectionConfigFactory.getInstance().getMibObjectList("default", ".1.9.9.9.9", "127.0.0.1", 0).size());
        for (MibObject object : DataCollectionConfigFactory.getInstance().getMibObjectList("default", ".1.3.6.1.4.1.200", "127.0.0.1", 0)) {
            assertEquals("Invalid MibObject: " + object, "ifIndex", object.getInstance());
        }
        assertArrayEquals(new String[0], DataCollectionConfigFactory.getInstance().getConfiguredResourceTypes().keySet().toArray(new String[0]));
    }

    @Test
    public void testNms6186() throws IOException {
        initDataCollectionFactory(m_xml_nms6186);
        assertEquals(0, DataCollectionConfigFactory.getInstance().getMibObjectList("default", ".1.9.9.9.9", "127.0.0.1", 0).size());
    }

    @Test
    public void testNms6186_2() throws IOException {
        String modifiedXml = m_xml_nms6186.replaceFirst("</snmp-collection>", m_systemsXmlFragment+"</snmp-collection>");
        initDataCollectionFactory(modifiedXml);
        assertEquals(12, DataCollectionConfigFactory.getInstance().getMibObjectList("default", ".1.3.6.1.4.1.200", "127.0.0.1", 0).size());
    }

    @Test
    public void testValidResourceType() throws IOException {
    	String modifiedXml = m_xml.replaceFirst("ifIndex", "brocadeIndex").replaceFirst("<groups", m_brocadeXmlFragment + "<groups");
        initDataCollectionFactory(modifiedXml);
        assertEquals(m_rrdRepository.getAbsolutePath(), DataCollectionConfigFactory.getInstance().getRrdPath());
        assertEquals(0, DataCollectionConfigFactory.getInstance().getMibObjectList("default", ".1.9.9.9.9", "127.0.0.1", 0).size());
        List<MibObject> mibObjects = DataCollectionConfigFactory.getInstance().getMibObjectList("default", ".1.3.6.1.4.1.200", "127.0.0.1", 0);
        // Make sure that the first value was edited as intended
        MibObject first = mibObjects.remove(0);
        assertEquals("Invalid MibObject: " + first, "brocadeIndex", first.getInstance());
        for (MibObject object : mibObjects) {
            assertEquals("Invalid MibObject: " + object, "ifIndex", object.getInstance());
        }
        assertArrayEquals(new String[] {"brocadeIndex"}, DataCollectionConfigFactory.getInstance().getConfiguredResourceTypes().keySet().toArray(new String[0]));
    }

    @Test
    public void testInvalidResourceType() throws IOException {
        String modifiedXml = m_xml.replaceFirst("ifIndex", "brocadeIndex");
        ThrowableAnticipator ta = new ThrowableAnticipator();
//        ta.anticipate(new DataAccessResourceFailureException("Instance 'brocadeIndex' invalid in mibObj definition for OID '.1.3.6.1.2.1.2.2.1.10' for group 'mib2-interfaces'. Allowable instance values: any positive number, 'ifIndex', or any of the custom resourceTypes."));
        ta.anticipate(new IllegalArgumentException("instance 'brocadeIndex' invalid in mibObj definition for OID '.1.3.6.1.2.1.2.2.1.10' in collection 'default' for group 'mib2-interfaces'.  Allowable instance values: any positive number, 'ifIndex', or any of the configured resourceTypes: (none)"));

        try {
            initDataCollectionFactory(modifiedXml);
            assertArrayEquals(new String[0], DataCollectionConfigFactory.getInstance().getConfiguredResourceTypes().keySet().toArray(new String[0]));
            DataCollectionConfigFactory.getInstance().getConfiguredResourceTypes();
        } catch (Throwable t) {
            ta.throwableReceived(t);
        }
        ta.verifyAnticipated();
    }

    @Test
    public void testMibObjProperties() throws Exception {
        String xml = "<datacollection-config rrdRepository = \"" + m_rrdRepository.getAbsolutePath() + File.separator + "\">" +
                "<snmp-collection name=\"default\" snmpStorageFlag=\"select\">" +
                "<rrd step=\"300\">" +
                "<rra>RRA:AVERAGE:0.5:1:8928</rra>" +
                "<rra>RRA:AVERAGE:0.5:12:8784</rra>" +
                "<rra>RRA:MIN:0.5:12:8784</rra>" +
                "<rra>RRA:MAX:0.5:12:8784</rra>" +
                "</rrd>" +
                "<resourceType name=\"bsnAPIfLoadParametersEntry\" label=\"Cisco Wireless AP Resources\" resourceLabel=\"${bsnAPName} (index ${index})\">" +
                "<persistenceSelectorStrategy class=\"org.opennms.netmgt.collection.support.PersistAllSelectorStrategy\"/>" +
                "<storageStrategy class=\"org.opennms.netmgt.collection.support.IndexStorageStrategy\"/>" +
                "</resourceType>" +
                "<groups>" +
                "<group name=\"bsnAPIfLoadParametersTable\" ifType=\"all\">" +
                "<mibObj oid=\".1.3.6.1.4.1.14179.2.2.13.1.4\" instance=\"bsnAPIfLoadParametersEntry\" alias=\"bsnAPIfLoadNumOfClients\" type=\"integer\" />" +
                "<property instance=\"bsnAPIfLoadParametersEntry\" alias=\"bsnAPName\">" +
                "<parameter key=\"source-type\" value=\"bsnAPEntry\" />" +
                "<parameter key=\"source-alias\" value=\"bsnAPName\" />" +
                "<parameter key=\"index-pattern\" value=\"^(.+)\\.\\d+$\"/>" +
                "</property>" +
                "</group>" +
                "</groups>" +
                "<systems>" +
                "<systemDef name=\"Test\">" +
                "<sysoidMask>.1.3.6.1.4.1.9999.</sysoidMask>" +
                "<collect>" +
                "<includeGroup>bsnAPIfLoadParametersTable</includeGroup>" +
                "</collect>" +
                "</systemDef>" +
                "</systems>" +
                "</snmp-collection>" +
                "</datacollection-config>";
        initDataCollectionFactory(xml);
        List<MibObjProperty> properties = DataCollectionConfigFactory.getInstance().getMibObjProperties("default", ".1.3.6.1.4.1.9999.1.1", null);
        assertEquals(1,  properties.size());
        assertEquals("bsnAPName", properties.get(0).getAlias());
        assertEquals("bsnAPEntry", properties.get(0).getParameterValue("source-type"));
    }

    private static void initDataCollectionFactory(String xmlConfig) {
        DefaultDataCollectionConfigDao dataCollectionDao = new DefaultDataCollectionConfigDao();
        dataCollectionDao.setConfigResource(new ByteArrayResource(xmlConfig.getBytes()));
        // Set the config directory to a blank value so that it doesn't pull in any extra config files
        dataCollectionDao.setConfigDirectory("");
        dataCollectionDao.afterPropertiesSet();
        DataCollectionConfigFactory.setInstance(dataCollectionDao);
    }

}
