/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.minion.status;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.opennms.core.criteria.Criteria;
import org.opennms.core.criteria.CriteriaBuilder;
import org.opennms.core.logging.Logging;
import org.opennms.minion.status.AggregateMinionStatus;
import org.opennms.minion.status.MinionStatus;
import org.opennms.netmgt.dao.api.MinionDao;
import org.opennms.netmgt.dao.api.NodeDao;
import org.opennms.netmgt.dao.api.OutageDao;
import org.opennms.netmgt.dao.api.ServiceTypeDao;
import org.opennms.netmgt.events.api.annotations.EventHandler;
import org.opennms.netmgt.events.api.annotations.EventListener;
import org.opennms.netmgt.events.api.model.IEvent;
import org.opennms.netmgt.model.OnmsMonitoringSystem;
import org.opennms.netmgt.model.OnmsNode;
import org.opennms.netmgt.model.minion.OnmsMinion;
import org.opennms.netmgt.model.outage.CurrentOutageDetails;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.TransactionCallback;
import org.springframework.transaction.support.TransactionCallbackWithoutResult;
import org.springframework.transaction.support.TransactionOperations;
import org.springframework.util.Assert;

@EventListener(name="minionStatusTracker", logPrefix="minion")
public class MinionStatusTracker
implements InitializingBean {
    private static final Logger LOG = LoggerFactory.getLogger(MinionStatusTracker.class);
    ScheduledExecutorService m_executor = Executors.newSingleThreadScheduledExecutor();
    public static final String LOG_PREFIX = "minion";
    private static final String OUTAGE_CREATED_EVENT_UEI = "uei.opennms.org/internal/poller/outageCreated";
    private static final String OUTAGE_RESOLVED_EVENT_UEI = "uei.opennms.org/internal/poller/outageResolved";
    static final String MINION_HEARTBEAT = "Minion-Heartbeat";
    static final String MINION_RPC = "Minion-RPC";
    @Autowired
    NodeDao m_nodeDao;
    @Autowired
    MinionDao m_minionDao;
    @Autowired
    ServiceTypeDao m_serviceTypeDao;
    @Autowired
    OutageDao m_outageDao;
    @Autowired
    TransactionOperations m_transactionOperations;
    private long m_refresh = TimeUnit.MINUTES.toMillis(5L);
    Map<Integer, OnmsMinion> m_minionNodes = new ConcurrentHashMap<Integer, OnmsMinion>();
    Map<String, OnmsMinion> m_minions = new ConcurrentHashMap<String, OnmsMinion>();
    Map<String, AggregateMinionStatus> m_state = new ConcurrentHashMap<String, AggregateMinionStatus>();

    public void afterPropertiesSet() throws Exception {
        try (Logging.MDCCloseable mdc = Logging.withPrefixCloseable((String)LOG_PREFIX);){
            LOG.info("Starting minion status tracker.");
            Assert.notNull((Object)this.m_nodeDao);
            Assert.notNull((Object)this.m_minionDao);
            Assert.notNull((Object)this.m_serviceTypeDao);
            Assert.notNull((Object)this.m_outageDao);
            Assert.notNull((Object)this.m_transactionOperations);
            Runnable command = new Runnable(){

                @Override
                public void run() {
                    try {
                        MinionStatusTracker.this.refresh();
                    }
                    catch (Throwable t) {
                        LOG.warn("Failed to refresh minion status from the database.", t);
                    }
                }
            };
            this.m_executor.scheduleAtFixedRate(command, 0L, this.m_refresh, TimeUnit.MILLISECONDS);
        }
    }

    public long getRefresh() {
        return this.m_refresh;
    }

    public void setRefresh(long refresh) {
        this.m_refresh = refresh;
    }

    public Collection<OnmsMinion> getMinions() {
        return this.m_minions.values();
    }

    public MinionStatus getStatus(String foreignId) {
        return this.m_state.get(foreignId);
    }

    public MinionStatus getStatus(OnmsMinion minion) {
        return this.m_state.get(minion.getId());
    }

    @EventHandler(uei="uei.opennms.org/internal/monitoringSystemAdded")
    public void onMonitoringSystemAdded(IEvent e) {
        this.runInLoggingTransaction(() -> {
            String id = e.getParm("monitoringSystemId").toString();
            LOG.debug("Monitoring system added: {}", (Object)id);
            if (id != null) {
                this.m_state.put(id, AggregateMinionStatus.up());
            }
        });
    }

    @EventHandler(uei="uei.opennms.org/internal/monitoringSystemDeleted")
    public void onMonitoringSystemDeleted(IEvent e) {
        this.runInLoggingTransaction(() -> {
            String id = e.getParm("monitoringSystemId").toString();
            if (id != null) {
                LOG.debug("Monitoring system removed: {}", (Object)id);
                OnmsMinion minion = this.m_minions.get(id);
                this.m_minions.remove(id);
                this.m_state.remove(id);
                if (minion != null) {
                    Iterator<Map.Entry<Integer, OnmsMinion>> it = this.m_minionNodes.entrySet().iterator();
                    while (it.hasNext()) {
                        Map.Entry<Integer, OnmsMinion> entry = it.next();
                        if (!entry.getValue().getId().equals(minion.getId())) continue;
                        it.remove();
                        break;
                    }
                }
            } else {
                LOG.warn("Monitoring system removed event received, but unable to determine ID: {}", (Object)e);
            }
        });
    }

    @EventHandler(uei="uei.opennms.org/nodes/nodeGainedService")
    public void onNodeGainedService(IEvent e) {
        if (!MINION_HEARTBEAT.equals(e.getService()) && !MINION_RPC.equals(e.getService())) {
            return;
        }
        this.runInLoggingTransaction(() -> {
            this.assertHasNodeId(e);
            Integer nodeId = e.getNodeid().intValue();
            OnmsMinion minion = this.getMinionForNodeId(nodeId);
            if (minion == null) {
                LOG.debug("No minion found for node ID {}", (Object)nodeId);
                return;
            }
            String minionId = minion.getId();
            LOG.debug("Node {}/{} gained a Minion service: {}", new Object[]{nodeId, minionId, e.getService()});
            AggregateMinionStatus state = this.m_state.get(minionId);
            if (state == null) {
                LOG.info("Found new Minion node: {}/{}", (Object)nodeId, (Object)minionId);
                AggregateMinionStatus aggregateMinionStatus = state = "down".equals(minion.getStatus()) ? AggregateMinionStatus.down() : AggregateMinionStatus.up();
            }
            if (MINION_HEARTBEAT.equals(e.getService())) {
                state = state.heartbeatUp();
            } else if (MINION_RPC.equals(e.getService())) {
                state = state.rpcUp();
            }
            this.updateStateIfChanged(minion, state, this.m_state.get(minionId));
        });
    }

    @EventHandler(uei="uei.opennms.org/nodes/nodeDeleted")
    public void onNodeDeleted(IEvent e) {
        this.runInLoggingTransaction(() -> {
            this.assertHasNodeId(e);
            Integer nodeId = e.getNodeid().intValue();
            OnmsMinion minion = this.getMinionForNodeId(nodeId);
            this.m_minionNodes.remove(nodeId);
            if (minion != null) {
                String minionId = minion.getId();
                LOG.debug("Minion node {}({}) deleted.", (Object)nodeId, (Object)minionId);
                this.updateStateIfChanged(minion, null, this.m_state.get(minionId));
                this.m_state.remove(minionId);
            }
        });
    }

    @EventHandler(ueis={"uei.opennms.org/internal/poller/outageCreated", "uei.opennms.org/internal/poller/outageResolved"})
    public void onOutageEvent(IEvent e) {
        boolean isPerspectiveNull;
        boolean isHeartbeat = MINION_HEARTBEAT.equals(e.getService());
        boolean isRpc = MINION_RPC.equals(e.getService());
        boolean bl = e.getParm("perspective") == null ? true : (isPerspectiveNull = e.getParm("perspective").getValue() == null);
        if (!isHeartbeat && !isRpc || !isPerspectiveNull) {
            return;
        }
        this.runInLoggingTransaction(() -> {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Minion {} service event received for node {}: {}", new Object[]{isHeartbeat ? "heartbeat" : "rpc", e.getNodeid(), e});
            }
            this.assertHasNodeId(e);
            OnmsMinion minion = this.getMinionForNodeId(e.getNodeid().intValue());
            String minionId = minion.getId();
            AggregateMinionStatus status = this.m_state.get(minionId);
            if (status == null) {
                status = AggregateMinionStatus.down();
            }
            String uei = e.getUei();
            if (MINION_HEARTBEAT.equalsIgnoreCase(e.getService())) {
                if (OUTAGE_CREATED_EVENT_UEI.equals(uei)) {
                    status = status.heartbeatDown();
                } else if (OUTAGE_RESOLVED_EVENT_UEI.equals(uei)) {
                    status = status.heartbeatUp();
                }
            } else if (MINION_RPC.equalsIgnoreCase(e.getService())) {
                if (OUTAGE_CREATED_EVENT_UEI.equals(uei)) {
                    status = status.rpcDown();
                } else if (OUTAGE_RESOLVED_EVENT_UEI.equals(uei)) {
                    status = status.rpcUp();
                }
            }
            this.updateStateIfChanged(minion, status, this.m_state.get(minionId));
        });
    }

    public void refresh() {
        this.runInLoggingTransaction(() -> {
            LOG.info("Refreshing minion status from the outages database.");
            ConcurrentHashMap<String, OnmsMinion> minions = new ConcurrentHashMap<String, OnmsMinion>();
            ConcurrentHashMap<Integer, OnmsMinion> minionNodes = new ConcurrentHashMap<Integer, OnmsMinion>();
            ConcurrentHashMap<String, AggregateMinionStatus> state = new ConcurrentHashMap<String, AggregateMinionStatus>();
            List dbMinions = this.m_minionDao.findAll();
            if (dbMinions.size() == 0) {
                LOG.info("No minions found in the database.  Skipping processing.  Next refresh in {} milliseconds.", (Object)this.m_refresh);
                return;
            }
            LOG.debug("Populating minion state from the database.  Found {} minions.", (Object)dbMinions.size());
            if (LOG.isDebugEnabled()) {
                LOG.debug("Processing minions: {}", dbMinions.stream().map(OnmsMonitoringSystem::getId).collect(Collectors.toList()));
            }
            AggregateMinionStatus upStatus = AggregateMinionStatus.up();
            dbMinions.forEach(minion -> {
                String minionId = minion.getId();
                minions.put(minionId, (OnmsMinion)minion);
            });
            Criteria c = new CriteriaBuilder(OnmsNode.class).in("foreignId", minions.keySet()).distinct().toCriteria();
            List nodes = this.m_nodeDao.findMatching(c);
            for (OnmsNode node2 : nodes) {
                this.m_nodeDao.initialize((Object)node2.getLocation());
            }
            LOG.debug("Mapping {} node IDs to minions.", (Object)nodes.size());
            if (LOG.isTraceEnabled()) {
                LOG.trace("Processing nodes: {}", nodes.stream().map(OnmsNode::getId).collect(Collectors.toList()));
            }
            nodes.forEach(node -> {
                OnmsMinion m = (OnmsMinion)minions.get(node.getForeignId());
                if (m.getLocation().equals(node.getLocation().getLocationName())) {
                    minionNodes.put(node.getId(), m);
                    state.put(node.getForeignId(), upStatus);
                }
            });
            Collection outages = this.m_outageDao.newestCurrentOutages(Arrays.asList(MINION_HEARTBEAT, MINION_RPC));
            if (outages != null && outages.size() > 0) {
                LOG.debug("Processing {} outage records.", (Object)outages.size());
                outages.stream().sorted(Comparator.comparing(CurrentOutageDetails::getOutageId).reversed()).forEach(outage -> {
                    AggregateMinionStatus existingStatus;
                    String foreignId = outage.getForeignId();
                    AggregateMinionStatus currentStatus = (AggregateMinionStatus)state.get(foreignId);
                    AggregateMinionStatus newStatus = this.transformStatus(currentStatus, outage.getServiceName(), null, outage.getIfLostService());
                    if (newStatus.equals(existingStatus = this.m_state.get(foreignId))) {
                        LOG.trace("{} status {} is unchanged.", (Object)foreignId, (Object)newStatus);
                    } else {
                        LOG.trace("{} status {} is different than {}, using it instead.", new Object[]{foreignId, newStatus, existingStatus});
                        state.put(foreignId, newStatus);
                    }
                });
            } else {
                LOG.debug("No minion-related outages were found.");
            }
            LOG.debug("Persisting states to the database.");
            minions.values().forEach(minion -> {
                AggregateMinionStatus oldState = this.m_state.get(minion.getId());
                AggregateMinionStatus newState = (AggregateMinionStatus)state.get(minion.getId());
                this.updateStateIfChanged((OnmsMinion)minion, newState, oldState);
            });
            this.m_state = state;
            this.m_minions = minions;
            this.m_minionNodes = minionNodes;
            LOG.info("Minion status updated from the outages database.  Next refresh in {} milliseconds.", (Object)this.m_refresh);
        });
    }

    private AggregateMinionStatus transformStatus(AggregateMinionStatus currentStatus, String serviceName, Date ifRegainedService, Date ifLostService) {
        AggregateMinionStatus newStatus;
        if (ifRegainedService != null) {
            if (MINION_HEARTBEAT.equals(serviceName)) {
                newStatus = currentStatus.heartbeatUp();
            } else if (MINION_RPC.equals(serviceName)) {
                newStatus = currentStatus.rpcUp();
            } else {
                LOG.warn("Unhandled 'up' outage record: service={}, lost={}, regained={}", new Object[]{serviceName, ifLostService, ifRegainedService});
                newStatus = currentStatus;
            }
        } else if (MINION_HEARTBEAT.equals(serviceName)) {
            newStatus = currentStatus.heartbeatDown();
        } else if (MINION_RPC.equals(serviceName)) {
            newStatus = currentStatus.rpcDown();
        } else {
            LOG.warn("Unhandled 'down' outage record: service={}, lost={}, regained={}", new Object[]{serviceName, ifLostService, ifRegainedService});
            newStatus = currentStatus;
        }
        return newStatus;
    }

    private void updateStateIfChanged(OnmsMinion minion, AggregateMinionStatus current, AggregateMinionStatus previous) {
        String newMinionStatus;
        String minionId = minion.getId();
        String currentMinionStatus = minion.getStatus();
        if (current == null) {
            LOG.debug("Minion {} does not have a state. This is likely because it does not have a monitored node in the 'Minions' requisition.", (Object)minionId);
            if (!"unknown".equals(currentMinionStatus)) {
                minion.setStatus("unknown");
                LOG.info("Minion {} status changed: {} -> {}", new Object[]{minionId, currentMinionStatus, minion.getStatus()});
                this.m_minionDao.saveOrUpdate((Object)minion);
            }
            this.m_state.remove(minionId);
            return;
        }
        this.m_state.put(minionId, current);
        String string = newMinionStatus = current.isUp() ? "up" : "down";
        if (newMinionStatus.equals(currentMinionStatus)) {
            LOG.trace("Minion {} status did not change: {} = {}", new Object[]{minionId, currentMinionStatus, newMinionStatus});
            return;
        }
        minion.setStatus(newMinionStatus);
        this.m_minionDao.saveOrUpdate((Object)minion);
        LOG.info("Minion {} status changed: {} -> {}", new Object[]{minionId, previous == null ? "Unknown" : previous.getState(), current.getState()});
        if (LOG.isDebugEnabled()) {
            LOG.debug("Minion {} status processed: Heartbeat: {} -> {}, RPC: {} -> {}", new Object[]{minionId, previous == null ? "Unknown" : previous.getHeartbeatStatus(), current.getHeartbeatStatus(), previous == null ? "Unknown" : previous.getRpcStatus(), current.getRpcStatus()});
        }
    }

    private OnmsMinion getMinionForNodeId(Integer nodeId) {
        if (this.m_minionNodes.containsKey(nodeId)) {
            return this.m_minionNodes.get(nodeId);
        }
        OnmsNode node = (OnmsNode)this.m_nodeDao.get((Serializable)nodeId);
        if (node == null) {
            IllegalStateException ex = new IllegalStateException("Unable to retrieve minion. The node (ID: " + nodeId + ") does not exist!");
            LOG.warn(ex.getMessage());
            throw ex;
        }
        this.m_nodeDao.initialize((Object)node.getLocation());
        String minionId = node.getForeignId();
        OnmsMinion minion = this.m_minionDao.findById(minionId);
        this.m_minionNodes.put(nodeId, minion);
        this.m_minions.put(minionId, minion);
        return minion;
    }

    private void assertHasNodeId(IEvent e) {
        if (e.getNodeid() == null || e.getNodeid() == 0L) {
            IllegalStateException ex = new IllegalStateException("Received a nodeGainedService event, but there is no node ID!");
            LOG.warn(ex.getMessage() + " {}", (Object)e, (Object)ex);
            throw ex;
        }
    }

    private void runInLoggingTransaction(final Runnable runnable) {
        this.m_transactionOperations.execute((TransactionCallback)new TransactionCallbackWithoutResult(){

            protected void doInTransactionWithoutResult(TransactionStatus status) {
                try (Logging.MDCCloseable mdc = Logging.withPrefixCloseable((String)MinionStatusTracker.LOG_PREFIX);){
                    runnable.run();
                    MinionStatusTracker.this.m_minionDao.flush();
                }
            }
        });
    }
}

