/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.web.notification;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.opennms.core.criteria.Criteria;
import org.opennms.core.spring.BeanUtils;
import org.opennms.core.utils.InetAddressUtils;
import org.opennms.netmgt.dao.api.AcknowledgmentDao;
import org.opennms.netmgt.dao.api.NotificationDao;
import org.opennms.netmgt.model.AckAction;
import org.opennms.netmgt.model.OnmsAcknowledgment;
import org.opennms.netmgt.model.OnmsCriteria;
import org.opennms.netmgt.model.OnmsNotification;
import org.opennms.netmgt.model.OnmsUserNotification;
import org.opennms.web.filter.Filter;
import org.opennms.web.notification.filter.NotificationCriteria;
import org.opennms.web.notification.filter.NotificationCriteria.NotificationCriteriaVisitor;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;

/**
 * <p>DaoWebNotificationRepository class.</p>
 *
 * @author ranger
 * @version $Id: $
 * @since 1.8.1
 */
public class DaoWebNotificationRepository implements WebNotificationRepository, InitializingBean {
    
    @Autowired
    NotificationDao m_notificationDao;
    
    @Autowired
    AcknowledgmentDao m_ackDao;
    
    @Override
    public void afterPropertiesSet() throws Exception {
        BeanUtils.assertAutowiring(this);
    }

    /**
     * TODO: Convert this function to use {@link Criteria} instead.
     */
    private static final OnmsCriteria getOnmsCriteria(final NotificationCriteria notificationCriteria){
        final OnmsCriteria criteria = new OnmsCriteria(OnmsNotification.class);
        criteria.createAlias("node", "node", OnmsCriteria.LEFT_JOIN);
        criteria.createAlias("serviceType", "serviceType", OnmsCriteria.LEFT_JOIN);
        criteria.createAlias("event", "event", OnmsCriteria.LEFT_JOIN);
        criteria.createAlias("event.distPoller", "distPoller", OnmsCriteria.LEFT_JOIN);
        
        notificationCriteria.visit(new NotificationCriteriaVisitor<RuntimeException>(){

            @Override
            public void visitAckType(AcknowledgeType ackType) throws RuntimeException {
                if(ackType == AcknowledgeType.ACKNOWLEDGED) {
                    criteria.add(Restrictions.isNotNull("answeredBy"));
                } else if (ackType == AcknowledgeType.UNACKNOWLEDGED) {
                   criteria.add(Restrictions.isNull("answeredBy")); 
                }
                // AcknowledgeType.BOTH just adds no restriction
            }

            @Override
            public void visitFilter(Filter filter) throws RuntimeException {
                criteria.add(filter.getCriterion());
                
            }

            @Override
            public void visitLimit(int limit, int offset) throws RuntimeException {
                criteria.setMaxResults(limit);
                criteria.setFirstResult(offset);                
            }

            @Override
            public void visitSortStyle(SortStyle sortStyle) throws RuntimeException {
                switch(sortStyle){
                    case LOCATION:
                        criteria.addOrder(Order.desc("event.distPoller.location"));
                        break;
                    case RESPONDER:
                        criteria.addOrder(Order.desc("answeredBy"));        
                        break;
                    case PAGETIME:
                        criteria.addOrder(Order.desc("pageTime"));
                        break;
                    case RESPONDTIME:
                        criteria.addOrder(Order.desc("respondTime"));
                        break;
                    case NODE:
                        criteria.addOrder(Order.desc("node.label"));
                        break;
                    case NODE_LOCATION:
                        criteria.addOrder(Order.desc("node.location.locationName"));
                        break;
                    case INTERFACE:
                        criteria.addOrder(Order.desc("ipAddress"));
                        break;
                    case SERVICE:
                        criteria.addOrder(Order.desc("serviceType.name"));
                        break;
                    case ID:
                        criteria.addOrder(Order.desc("notifyId"));
                        break;
                    case SEVERITY:
                        criteria.addOrder(Order.desc("event.eventSeverity"));
                        break;
                    case REVERSE_LOCATION:
                        criteria.addOrder(Order.desc("event.distPoller.location"));
                        break;
                    case REVERSE_RESPONDER:
                        criteria.addOrder(Order.asc("answeredBy"));            
                        break;
                    case REVERSE_PAGETIME:
                        criteria.addOrder(Order.asc("pageTime"));
                        break;
                    case REVERSE_RESPONDTIME:
                        criteria.addOrder(Order.asc("respondTime"));
                        break;
                    case REVERSE_NODE:
                        criteria.addOrder(Order.asc("node.label"));
                        break;
                    case REVERSE_NODE_LOCATION:
                        criteria.addOrder(Order.asc("node.location.locationName"));
                        break;
                    case REVERSE_INTERFACE:
                        criteria.addOrder(Order.asc("ipAddress"));
                        break;
                    case REVERSE_SERVICE:
                        criteria.addOrder(Order.asc("serviceType.name"));
                        break;
                    case REVERSE_ID:
                        criteria.addOrder(Order.asc("notifyId"));
                        break;
                    case REVERSE_SEVERITY:
                        criteria.addOrder(Order.asc("event.eventSeverity"));
                        break;
                    
                }
                
            }
            
        });
        
        return criteria;
    }

    private static Notification mapOnmsNotificationToNotification(OnmsNotification onmsNotification){
        if(onmsNotification != null){
            Notification notif = new Notification();
            notif.m_eventId = onmsNotification.getEvent() != null ? onmsNotification.getEvent().getId() : 0;
            notif.m_interfaceID = onmsNotification.getIpAddress() == null ? null : InetAddressUtils.toIpAddrString(onmsNotification.getIpAddress());
            notif.m_nodeID = onmsNotification.getNode() != null ? onmsNotification.getNode().getId() : 0;
            notif.m_notifyID = onmsNotification.getNotifyId();
            notif.m_numMsg = onmsNotification.getNumericMsg();
            notif.m_responder = onmsNotification.getAnsweredBy();
            notif.m_serviceId = onmsNotification.getServiceType() != null ? onmsNotification.getServiceType().getId() : 0;
            notif.m_serviceName = onmsNotification.getServiceType() != null ? onmsNotification.getServiceType().getName() : "";
            notif.m_timeReply = onmsNotification.getRespondTime() != null ? onmsNotification.getRespondTime().getTime() : 0;
            notif.m_timeSent = onmsNotification.getPageTime() != null ? onmsNotification.getPageTime().getTime() : 0;
            notif.m_txtMsg = onmsNotification.getTextMsg();

            // Add the OnmsUserNotifications as NoticeSentTo instances
            final List<NoticeSentTo> sentToList = new ArrayList<>();
            for (OnmsUserNotification userNotified : onmsNotification.getUsersNotified()) {
                NoticeSentTo newSentTo = new NoticeSentTo();
                newSentTo.setUserId(userNotified.getUserId());
                // Can be null
                if (userNotified.getNotifyTime() == null) {
                    newSentTo.setTime(0);
                } else {
                    newSentTo.setTime(userNotified.getNotifyTime().getTime());
                }
                // Can be null
                newSentTo.setMedia(userNotified.getMedia());
                // Can be null
                newSentTo.setContactInfo(userNotified.getContactInfo());
                sentToList.add(newSentTo);
            }
            notif.m_sentTo = sentToList;

            return notif;
        }else{
            return null;
        }
    }
    
    /** {@inheritDoc} */
    @Transactional
    @Override
    public void acknowledgeMatchingNotification(String user, Date timestamp, NotificationCriteria criteria) {
        List<OnmsNotification> notifs = m_notificationDao.findMatching(getOnmsCriteria(criteria));
        
        for (OnmsNotification notif : notifs) {
            
            OnmsAcknowledgment ack = new OnmsAcknowledgment(notif, user);
            ack.setAckAction(AckAction.ACKNOWLEDGE);
            ack.setAckTime(timestamp);
            m_ackDao.processAck(ack);
        }
    }
    
    /** {@inheritDoc} */
    @Transactional
    @Override
    public int countMatchingNotifications(NotificationCriteria criteria) {
        return m_notificationDao.countMatching(getOnmsCriteria(criteria));
    }

    /** {@inheritDoc} */
    @Transactional
    @Override
    public Notification[] getMatchingNotifications(NotificationCriteria criteria) {
        List<Notification> notifications = new ArrayList<>();
        List<OnmsNotification> onmsNotifs = m_notificationDao.findMatching(getOnmsCriteria(criteria));

        for (OnmsNotification notif : onmsNotifs) {
            notifications.add(mapOnmsNotificationToNotification(notif));
        }
        
        return notifications.toArray(new Notification[0]);
    }
    
    /** {@inheritDoc} */
    @Transactional
    @Override
    public Notification getNotification(int noticeId) {
        return mapOnmsNotificationToNotification(m_notificationDao.get(noticeId));
    }
}
