/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.config.snmpinterfacepoller;


import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import org.opennms.core.xml.ValidateUsing;
import org.opennms.netmgt.config.utils.ConfigUtils;

/**
 * Configuration of node-outage functionality
 */
@XmlRootElement(name = "node-outage")
@XmlAccessorType(XmlAccessType.FIELD)
@ValidateUsing("snmp-interface-poller-configuration.xsd")
public class NodeOutage implements Serializable {
    private static final long serialVersionUID = 1L;

    /**
     * Critical service. Defining a critical service greatly
     *  reduces the traffic generated by the snmp poller when a service is DOWN.
     *  The snmp poller stop polling if one of the critical service is down on the
     * 
     *  SNMP primary node interface.
     */
    @XmlElement(name = "critical-service", required = true)
    private List<CriticalService> m_criticalServices = new ArrayList<>();

    public NodeOutage() {
    }

    public List<CriticalService> getCriticalServices() {
        return m_criticalServices;
    }

    public void setCriticalServices(final List<CriticalService> criticalServices) {
        ConfigUtils.assertMinimumSize(criticalServices, 1, "critical-service");
        if (criticalServices == m_criticalServices) return;
        m_criticalServices.clear();
        if (criticalServices != null) m_criticalServices.addAll(criticalServices);
    }

    public void addCriticalService(final CriticalService criticalService) {
        m_criticalServices.add(criticalService);
    }

    public boolean removeCriticalService(final CriticalService criticalService) {
        return m_criticalServices.remove(criticalService);
    }

    @Override
    public int hashCode() {
        return Objects.hash(m_criticalServices);
    }

    @Override
    public boolean equals(final Object obj) {
        if ( this == obj ) {
            return true;
        }

        if (obj instanceof NodeOutage) {
            final NodeOutage that = (NodeOutage)obj;
            return Objects.equals(this.m_criticalServices, that.m_criticalServices);
        }
        return false;
    }

}
