/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.config.notifd;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import org.opennms.core.xml.ValidateUsing;
import org.opennms.netmgt.config.utils.ConfigUtils;

/**
 * Top-level element for the notifd-configuration.xml
 *  configuration file.
 */
@XmlRootElement(name = "notifd-configuration")
@XmlAccessorType(XmlAccessType.FIELD)
@ValidateUsing("notifd-configuration.xsd")
public class NotifdConfiguration implements Serializable {
    private static final long serialVersionUID = 2L;

    private static final String DEFAULT_PAGES_SENT = "SELECT * FROM notifications";
    private static final String DEFAULT_NEXT_NOTIFID = "SELECT nextval('notifynxtid')";
    private static final String DEFAULT_NEXT_USER_NOTIFID = "SELECT nextval('userNotifNxtId')";
    private static final String DEFAULT_NEXT_GROUP_ID = "SELECT nextval('notifygrpid')";
    private static final String DEFAULT_SERVICEID_SQL = "SELECT serviceID from service where serviceName = ?";
    private static final String DEFAULT_OUTSTANDING_NOTICES_SQL = "SELECT notifyid FROM notifications where notifyId = ? AND respondTime is not null";
    private static final String DEFAULT_ACKNOWLEDGEID_SQL = "SELECT notifyid FROM notifications WHERE eventuei=? AND nodeid=? AND interfaceid=? AND serviceid=?";
    private static final String DEFAULT_ACKNOWLEDGE_UPDATE_SQL = "UPDATE notifications SET answeredby=?, respondtime=? WHERE notifyId=?";
    private static final String DEFAULT_EMAIL_ADDRESS_COMMAND = "javaEmail";
    private static final Integer DEFAULT_MAX_THREADS = 100;

    @XmlAttribute(name = "status", required = true)
    private String m_status;

    @XmlAttribute(name = "pages-sent")
    private String m_pagesSent;

    @XmlAttribute(name = "next-notif-id")
    private String m_nextNotifId;

    @XmlAttribute(name = "next-user-notif-id")
    private String m_nextUserNotifId;

    @XmlAttribute(name = "next-group-id")
    private String m_nextGroupId;

    @XmlAttribute(name = "service-id-sql")
    private String m_serviceIdSql;

    @XmlAttribute(name = "outstanding-notices-sql")
    private String m_outstandingNoticesSql;

    @XmlAttribute(name = "acknowledge-id-sql")
    private String m_acknowledgeIdSql;

    @XmlAttribute(name = "acknowledge-update-sql")
    private String m_acknowledgeUpdateSql;

    @XmlAttribute(name = "match-all", required = true)
    private Boolean m_matchAll;

    @XmlAttribute(name = "email-address-command")
    private String m_emailAddressCommand;

    @XmlAttribute(name = "numeric-skip-resolution-prefix")
    private Boolean m_numericSkipResolutionPrefix;

    @XmlAttribute(name = "max-threads")
    private Integer m_maxThreads;

    @XmlElement(name = "auto-acknowledge-alarm")
    private AutoAcknowledgeAlarm m_autoAcknowledgeAlarm;

    @XmlElement(name = "auto-acknowledge")
    private List<AutoAcknowledge> m_autoAcknowledges = new ArrayList<>();

    @XmlElement(name = "queue", required = true)
    private List<Queue> m_queues = new ArrayList<>();

    @XmlElement(name = "outage-calendar")
    private List<String> m_outageCalendars = new ArrayList<>();

    public NotifdConfiguration() { }

    public String getStatus() {
        return m_status;
    }

    public void setStatus(final String status) {
        m_status = ConfigUtils.assertNotEmpty(status, "status");
    }

    public String getPagesSent() {
        return m_pagesSent != null ? m_pagesSent : DEFAULT_PAGES_SENT;
    }

    public void setPagesSent(final String pagesSent) {
        m_pagesSent = ConfigUtils.normalizeString(pagesSent);
    }

    public String getNextNotifId() {
        return m_nextNotifId != null ? m_nextNotifId : DEFAULT_NEXT_NOTIFID;
    }

    public void setNextNotifId(final String nextNotifId) {
        m_nextNotifId = ConfigUtils.normalizeString(nextNotifId);
    }

    public String getNextUserNotifId() {
        return m_nextUserNotifId != null ? m_nextUserNotifId : DEFAULT_NEXT_USER_NOTIFID;
    }

    public void setNextUserNotifId(final String nextUserNotifId) {
        m_nextUserNotifId = ConfigUtils.normalizeString(nextUserNotifId);
    }

    public String getNextGroupId() {
        return m_nextGroupId != null ? m_nextGroupId : DEFAULT_NEXT_GROUP_ID;
    }

    public void setNextGroupId(final String nextGroupId) {
        m_nextGroupId = ConfigUtils.normalizeString(nextGroupId);
    }

    public String getServiceIdSql() {
        return m_serviceIdSql != null ? m_serviceIdSql : DEFAULT_SERVICEID_SQL;
    }

    public void setServiceIdSql(final String serviceIdSql) {
        m_serviceIdSql = ConfigUtils.normalizeString(serviceIdSql);
    }

    public String getOutstandingNoticesSql() {
        return m_outstandingNoticesSql != null ? m_outstandingNoticesSql : DEFAULT_OUTSTANDING_NOTICES_SQL;
    }

    public void setOutstandingNoticesSql(final String outstandingNoticesSql) {
        m_outstandingNoticesSql = ConfigUtils.normalizeString(outstandingNoticesSql);
    }

    public String getAcknowledgeIdSql() {
        return m_acknowledgeIdSql != null ? m_acknowledgeIdSql : DEFAULT_ACKNOWLEDGEID_SQL;
    }

    public void setAcknowledgeIdSql(final String acknowledgeIdSql) {
        m_acknowledgeIdSql = ConfigUtils.normalizeString(acknowledgeIdSql);
    }

    public String getAcknowledgeUpdateSql() {
        return m_acknowledgeUpdateSql != null ? m_acknowledgeUpdateSql : DEFAULT_ACKNOWLEDGE_UPDATE_SQL;
    }

    public void setAcknowledgeUpdateSql(final String acknowledgeUpdateSql) {
        m_acknowledgeUpdateSql = ConfigUtils.normalizeString(acknowledgeUpdateSql);
    }

    public Boolean getMatchAll() {
        return m_matchAll;
    }

    public void setMatchAll(final Boolean matchAll) {
        m_matchAll = ConfigUtils.assertNotNull(matchAll, "match-all");
    }

    public String getEmailAddressCommand() {
        return m_emailAddressCommand != null ? m_emailAddressCommand : DEFAULT_EMAIL_ADDRESS_COMMAND;
    }

    public void setEmailAddressCommand(final String emailAddressCommand) {
        m_emailAddressCommand = ConfigUtils.normalizeString(emailAddressCommand);
    }

    public Boolean getNumericSkipResolutionPrefix() {
        return m_numericSkipResolutionPrefix != null ? m_numericSkipResolutionPrefix : Boolean.valueOf("false");
    }

    public void setNumericSkipResolutionPrefix(final Boolean prefix) {
        m_numericSkipResolutionPrefix = prefix;
    }

    public Integer getMaxThreads() {
        return m_maxThreads != null ? m_maxThreads : DEFAULT_MAX_THREADS;
    }

    public void setMaxThreads(Integer maxThreads) {
        m_maxThreads = maxThreads;
    }

    public Optional<AutoAcknowledgeAlarm> getAutoAcknowledgeAlarm() {
        return Optional.ofNullable(m_autoAcknowledgeAlarm);
    }

    public void setAutoAcknowledgeAlarm(final AutoAcknowledgeAlarm autoAcknowledgeAlarm) {
        m_autoAcknowledgeAlarm = autoAcknowledgeAlarm;
    }

    public List<AutoAcknowledge> getAutoAcknowledges() {
        return m_autoAcknowledges;
    }

    public void setAutoAcknowledges(final List<AutoAcknowledge> autoAcknowledges) {
        m_autoAcknowledges.clear();
        m_autoAcknowledges.addAll(autoAcknowledges);
    }

    public void addAutoAcknowledge(final AutoAcknowledge autoAcknowledge) throws IndexOutOfBoundsException {
        m_autoAcknowledges.add(autoAcknowledge);
    }

    public List<String> getOutageCalendars() {
        return m_outageCalendars;
    }

    public void setOutageCalendars(final List<String> calendars) {
        if (calendars == m_outageCalendars) return;
        m_outageCalendars.clear();
        if (calendars != null) m_outageCalendars.addAll(calendars);
    }

    public void addOutageCalendar(final String calendar) {
        m_outageCalendars.add(calendar);
    }

    public boolean removeOutageCalendar(final String calendar) {
        return m_outageCalendars.remove(calendar);
    }

    public List<Queue> getQueues() {
        return m_queues;
    }

    public void setQueues(final List<Queue> queues) {
        if (queues == m_queues) return;
        m_queues.clear();
        if (queues != null) m_queues.addAll(queues);
    }

    public void addQueue(final Queue queue) {
        m_queues.add(queue);
    }

    @Override
    public int hashCode() {
        return Objects.hash(m_status, 
                            m_pagesSent, 
                            m_nextNotifId, 
                            m_nextUserNotifId, 
                            m_nextGroupId, 
                            m_serviceIdSql, 
                            m_outstandingNoticesSql, 
                            m_acknowledgeIdSql, 
                            m_acknowledgeUpdateSql, 
                            m_matchAll, 
                            m_emailAddressCommand, 
                            m_numericSkipResolutionPrefix,
                            m_maxThreads,
                            m_autoAcknowledgeAlarm, 
                            m_autoAcknowledges, 
                            m_queues, 
                            m_outageCalendars);
    }

    @Override
    public boolean equals(final Object obj) {
        if ( this == obj ) {
            return true;
        }

        if (obj instanceof NotifdConfiguration) {
            final NotifdConfiguration that = (NotifdConfiguration)obj;
            return Objects.equals(this.m_status, that.m_status)
                    && Objects.equals(this.m_pagesSent, that.m_pagesSent)
                    && Objects.equals(this.m_nextNotifId, that.m_nextNotifId)
                    && Objects.equals(this.m_nextUserNotifId, that.m_nextUserNotifId)
                    && Objects.equals(this.m_nextGroupId, that.m_nextGroupId)
                    && Objects.equals(this.m_serviceIdSql, that.m_serviceIdSql)
                    && Objects.equals(this.m_outstandingNoticesSql, that.m_outstandingNoticesSql)
                    && Objects.equals(this.m_acknowledgeIdSql, that.m_acknowledgeIdSql)
                    && Objects.equals(this.m_acknowledgeUpdateSql, that.m_acknowledgeUpdateSql)
                    && Objects.equals(this.m_matchAll, that.m_matchAll)
                    && Objects.equals(this.m_emailAddressCommand, that.m_emailAddressCommand)
                    && Objects.equals(this.m_numericSkipResolutionPrefix, that.m_numericSkipResolutionPrefix)
                    && Objects.equals(this.m_maxThreads, that.m_maxThreads)
                    && Objects.equals(this.m_autoAcknowledgeAlarm, that.m_autoAcknowledgeAlarm)
                    && Objects.equals(this.m_autoAcknowledges, that.m_autoAcknowledges)
                    && Objects.equals(this.m_queues, that.m_queues)
                    && Objects.equals(this.m_outageCalendars, that.m_outageCalendars);
        }
        return false;
    }

}
