/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.config;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.opennms.core.test.xml.XmlTest.assertXmlEquals;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.junit.Test;
import org.opennms.core.network.IPAddress;
import org.opennms.core.network.IPAddressRange;
import org.opennms.core.network.IPAddressRangeSet;
import org.opennms.netmgt.config.snmp.Definition;
import org.opennms.netmgt.config.snmp.SnmpProfile;
import org.opennms.netmgt.events.api.EventConstants;
import org.opennms.netmgt.events.api.model.ImmutableMapper;
import org.opennms.netmgt.snmp.SnmpAgentConfig;
import org.opennms.netmgt.snmp.SnmpConfiguration;
import org.opennms.netmgt.xml.event.Event;
import org.opennms.netmgt.xml.event.Parm;
import org.springframework.core.io.AbstractResource;

/**
 * JUnit tests for the configureSNMP event handling and optimization of
 * the SNMP configuration XML.
 * 
 * @author <a href="mailto:david@opennms.org>David Hustace</a>
 *
 */
public class SnmpEventInfoTest {

    @Test(expected=IllegalArgumentException.class)
    public void testConfigRangeCreateOutOfOrder() {
        new IPAddressRange("192.168.1.2", "192.168.1.1");
    }
    
    @Test
    public void testAddressIncr() {
        IPAddress ipA = new IPAddress("::5");
        assertEquals("::6", ipA.incr().toString());
        
        IPAddress ipB = new IPAddress("::ffff:ffff");
        assertEquals("::1:0:0", ipB.incr().toString());
        
        IPAddress ipC = new IPAddress("::ff00:ffff");
        assertEquals("::ff01:0", ipC.incr().toString());

        IPAddress ipD = new IPAddress("::ff00:7fff");
        assertEquals("::ff00:8000", ipD.incr().toString());

    }
    
    @Test
    public void testConfigAddressDecr() {
        IPAddress a = new IPAddress("::6");
        assertEquals("::5", a.decr().toString());
        
        IPAddress b = new IPAddress("::1:0:0");
        assertEquals("::ffff:ffff", b.decr().toString());
        
        IPAddress c = new IPAddress("ff::ffff:1:0");
        assertEquals("ff::ffff:0:ffff", c.decr().toString());

        IPAddress d = new IPAddress("ff::ffff:1:8000");
        assertEquals("ff::ffff:1:7fff", d.decr().toString());
    }

    @Test
    public void testContainsAddr() {
        IPAddressRange range = new IPAddressRange("192.168.1.1", "192.168.1.3");
        assertFalse(range.contains("192.168.0.1"));
        assertTrue(range.contains("192.168.1.1"));
        assertTrue(range.contains("192.168.1.2"));
        assertTrue(range.contains("192.168.1.3"));
        assertFalse(range.contains("192.168.1.4"));
    }

    @Test
    public void testFollows() {
        IPAddressRange s = new IPAddressRange("192.168.1.5", "192.168.1.6");
        IPAddressRange q = new IPAddressRange("192.168.1.1", "192.168.1.2");
        IPAddressRange r = new IPAddressRange("192.168.1.3", "192.168.1.4");
        assertTrue(r.comesAfter(q));
        assertFalse(r.comesAfter(r));
        assertFalse(r.comesAfter(s));
    }

    @Test
    public void testContainsAddrIPv6() {
        IPAddressRange r = new IPAddressRange("2001:db8::10", "2001:db8::20");
        assertFalse(r.contains("192.168.0.1"));
        assertFalse(r.contains("2001:db8::1"));
        assertTrue(r.contains("2001:db8::10"));
        assertTrue(r.contains("2001:db8::15"));
        assertTrue(r.contains("2001:db8::20"));
        assertFalse(r.contains("2001:db8::21"));
    }


    @Test
    public void testContainsRange() {
        IPAddressRange r = new IPAddressRange("192.168.1.1", "192.168.1.10");
        assertTrue(r.contains(new IPAddressRange("192.168.1.1", "192.168.1.1")));
        assertTrue(r.contains(new IPAddressRange("192.168.1.10", "192.168.1.10")));
        assertTrue(r.contains(new IPAddressRange("192.168.1.2", "192.168.1.7")));
        assertFalse(r.contains(new IPAddressRange("192.168.1.0", "192.168.1.1")));
        assertFalse(r.contains(new IPAddressRange("192.168.1.2", "192.168.1.11")));
        assertFalse(r.contains(new IPAddressRange("192.168.1.0", "192.168.1.11")));

    }

    @Test
    public void testPreceedsRange() {
        IPAddressRange r = new IPAddressRange("192.168.1.10", "192.168.1.20");
        assertTrue(r.comesBefore(new IPAddressRange("192.168.1.21", "192.168.1.30")));
        assertTrue(r.comesBefore(new IPAddressRange("192.168.1.21", "192.168.1.21")));
        assertFalse(r.comesBefore(new IPAddressRange("192.168.1.20", "192.168.1.30")));
        assertFalse(r.comesBefore(new IPAddressRange("192.168.1.7", "192.168.1.9")));
    }

    @Test
    public void testOverlapsRange() {
        IPAddressRange r = new IPAddressRange("192.168.1.10", "192.168.1.20");
        assertTrue(r.overlaps(new IPAddressRange("192.168.1.10", "192.168.1.10")));
        assertTrue(r.overlaps(new IPAddressRange("192.168.1.20", "192.168.1.20")));
        assertTrue(r.overlaps(new IPAddressRange("192.168.1.10", "192.168.1.20")));
        assertTrue(r.overlaps(new IPAddressRange("192.168.1.9", "192.168.1.22")));
        assertTrue(r.overlaps(new IPAddressRange("192.168.1.15", "192.168.1.22")));
        assertTrue(r.overlaps(new IPAddressRange("192.168.1.9", "192.168.1.15")));
        assertTrue(r.overlaps(new IPAddressRange("192.168.1.11", "192.168.1.19")));
        assertFalse(r.overlaps(new IPAddressRange("192.168.1.5", "192.168.1.9")));
        assertFalse(r.overlaps(new IPAddressRange("192.168.1.21", "192.168.1.22")));

    }

    @Test
    public void testAdjacentRange() {
        IPAddressRange r = new IPAddressRange("192.168.1.10", "192.168.1.20");
        assertTrue(r.adjoins(new IPAddressRange("192.168.1.7", "192.168.1.9")));
        assertFalse(r.adjoins(new IPAddressRange("192.168.1.7", "192.168.1.8")));
        assertFalse(r.adjoins(new IPAddressRange("192.168.1.7", "192.168.1.21")));
        assertTrue(r.adjoins(new IPAddressRange("192.168.1.21", "192.168.1.21")));
    }
    
    @Test
    public void testConfigRangeEquals() {
        IPAddressRange r = new IPAddressRange("192.168.1.10", "192.168.1.20");
        assertEquals(r, r);
        assertEquals(r, new IPAddressRange("192.168.1.10", "192.168.1.20"));
        assertFalse(r.equals(new IPAddressRange("192.168.1.10", "192.168.1.19")));
    }
    
    @Test
    public void testCombine() {
        IPAddressRange rr = new IPAddressRange("192.168.1.10", "192.168.1.20");
        assertEquals(rr, rr.combine(new IPAddressRange("192.168.1.10", "192.168.1.20")));
        assertEquals(rr, rr.combine(new IPAddressRange("192.168.1.11", "192.168.1.20")));
        assertEquals(rr, rr.combine(new IPAddressRange("192.168.1.10", "192.168.1.19")));
        assertEquals(rr, rr.combine(new IPAddressRange("192.168.1.11", "192.168.1.19")));
        assertEquals(new IPAddressRange("192.168.1.9", "192.168.1.20"), rr.combine(new IPAddressRange("192.168.1.9", "192.168.1.12")));
        assertEquals(new IPAddressRange("192.168.1.10", "192.168.1.22"), rr.combine(new IPAddressRange("192.168.1.13", "192.168.1.22")));
        assertEquals(new IPAddressRange("192.168.1.9", "192.168.1.22"), rr.combine(new IPAddressRange("192.168.1.9", "192.168.1.22")));
        assertEquals(new IPAddressRange("192.168.1.7", "192.168.1.20"), rr.combine(new IPAddressRange("192.168.1.7", "192.168.1.9")));
        assertEquals(new IPAddressRange("192.168.1.10", "192.168.1.24"), rr.combine(new IPAddressRange("192.168.1.21", "192.168.1.24")));
    }
    
    @Test
    public void testRemove() {
        IPAddressRange r = new IPAddressRange("192.168.1.10", "192.168.1.20");
        assertArrayEquals(new IPAddressRange[0], r.remove(r));
        assertArrayEquals(new IPAddressRange[0], r.remove(new IPAddressRange("192.168.1.5", "192.168.1.27")));
        assertArrayEquals(new IPAddressRange[] { r }, r.remove(new IPAddressRange("192.168.1.5", "192.168.1.7")));
        assertArrayEquals(new IPAddressRange[] { r }, r.remove(new IPAddressRange("192.168.1.22", "192.168.1.27")));
        assertArrayEquals(new IPAddressRange[] { new IPAddressRange("192.168.1.15", "192.168.1.20") }, r.remove(new IPAddressRange("192.168.1.5", "192.168.1.14")));
        assertArrayEquals(new IPAddressRange[] { new IPAddressRange("192.168.1.10", "192.168.1.14") }, r.remove(new IPAddressRange("192.168.1.15", "192.168.1.24")));
        assertArrayEquals(new IPAddressRange[] { new IPAddressRange("192.168.1.10", "192.168.1.14"), new IPAddressRange("192.168.1.16", "192.168.1.20") }, r.remove(new IPAddressRange("192.168.1.15", "192.168.1.15")));
    }
    
    @Test
    public void testListAdd() {
        IPAddressRange r = new IPAddressRange("192.168.1.10", "192.168.1.20");

        IPAddressRangeSet ranges = new IPAddressRangeSet();
        ranges.add(r);
        
        assertArrayEquals(new IPAddressRange[] { r }, ranges.toArray());
        
        IPAddressRange s = new IPAddressRange("192.168.1.30", "192.168.1.40");
        ranges.add(s);

        assertArrayEquals(new IPAddressRange[] { r, s }, ranges.toArray());

        IPAddressRange t = new IPAddressRange("192.168.1.2", "192.168.1.8");
        ranges.add(t);

        assertArrayEquals(new IPAddressRange[] { t, r, s }, ranges.toArray());

        IPAddressRange u = new IPAddressRange("192.168.1.22", "192.168.1.28");
        ranges.add(u);

        assertArrayEquals(new IPAddressRange[] { t, r, u, s }, ranges.toArray());
        
        ranges.add(new IPAddressRange("192.168.1.18", "192.168.1.24"));
        
        assertArrayEquals(new IPAddressRange[] { t, new IPAddressRange("192.168.1.10", "192.168.1.28"), s }, ranges.toArray());

        ranges.add(new IPAddressRange("192.168.1.9", "192.168.1.9"));
        
        assertArrayEquals(new IPAddressRange[] { new IPAddressRange("192.168.1.2", "192.168.1.28"), s }, ranges.toArray());

    }
    
    @Test
    public void testListRemove() {
        IPAddressRange r = new IPAddressRange("192.168.1.1", "192.168.1.100"); //{[1..100]}

        IPAddressRangeSet ranges = new IPAddressRangeSet();
        ranges.add(r);
        
        IPAddressRange s = new IPAddressRange("192.168.1.30", "192.168.1.40"); 
        ranges.remove(s); //{[1..29],[41..100]}

        assertArrayEquals(new IPAddressRange[] { new IPAddressRange("192.168.1.1", "192.168.1.29"), new IPAddressRange("192.168.1.41", "192.168.1.100") }, ranges.toArray());

        IPAddressRange t = new IPAddressRange("192.168.1.20", "192.168.1.35"); 
        ranges.remove(t); //{[1..19],[41..100]}

        assertArrayEquals(new IPAddressRange[] { new IPAddressRange("192.168.1.1", "192.168.1.19"), new IPAddressRange("192.168.1.41", "192.168.1.100") }, ranges.toArray());

        IPAddressRange u = new IPAddressRange("192.168.1.35", "192.168.1.50"); 
        ranges.remove(u); //{[1..19],[51..100]}

        assertArrayEquals(new IPAddressRange[] { new IPAddressRange("192.168.1.1", "192.168.1.19"), new IPAddressRange("192.168.1.51", "192.168.1.100") }, ranges.toArray());
        
        ranges.remove(new IPAddressRange("192.168.1.60", "192.168.1.70")); //{1..19],[51..59],[71..100]}
        
        assertArrayEquals(new IPAddressRange[] { new IPAddressRange("192.168.1.1", "192.168.1.19"), new IPAddressRange("192.168.1.51", "192.168.1.59"), new IPAddressRange("192.168.1.71", "192.168.1.100") }, ranges.toArray());

        ranges.remove(new IPAddressRange("192.168.1.10", "192.168.1.80")); //{1..9],[81..100}
        
        assertArrayEquals(new IPAddressRange[] { new IPAddressRange("192.168.1.1", "192.168.1.9"), new IPAddressRange("192.168.1.81", "192.168.1.100") }, ranges.toArray());

    }

    @Test
    public void testCreateDefForSnmpV1() {
    	SnmpEventInfo snmpEventInfo = new SnmpEventInfo();
    	snmpEventInfo.setAuthPassPhrase("authPassPhrase");
    	snmpEventInfo.setAuthProtocol("authProtocol");
    	snmpEventInfo.setReadCommunityString("readCommunityString");
    	snmpEventInfo.setWriteCommunityString("writeCommunityString");
    	snmpEventInfo.setContextEngineId("contextEngineId");
    	snmpEventInfo.setContextName("contextName");
    	snmpEventInfo.setEngineId("engineId");
    	snmpEventInfo.setEnterpriseId("enterpriseId");
    	snmpEventInfo.setMaxRepetitions(1000);
    	snmpEventInfo.setMaxVarsPerPdu(2000);
    	snmpEventInfo.setMaxRequestSize(7000);
    	snmpEventInfo.setPort(3000);
    	snmpEventInfo.setPrivPassPhrase("privPassPhrase");
    	snmpEventInfo.setPrivProtocol("privProtocol");
    	snmpEventInfo.setProxyHost("127.0.0.1");
    	snmpEventInfo.setRetryCount(4000);
    	snmpEventInfo.setSecurityLevel(5000);
    	snmpEventInfo.setTimeout(6000);
    	snmpEventInfo.setVersion("v1");
    	
    	// check v1/commons properties
    	Definition def = snmpEventInfo.createDef();
    	assertEquals(snmpEventInfo.getReadCommunityString(), def.getReadCommunity());
    	assertEquals(snmpEventInfo.getMaxRepetitions(), def.getMaxRepetitions().intValue());
    	assertEquals(snmpEventInfo.getMaxVarsPerPdu(), def.getMaxVarsPerPdu().intValue());
    	assertEquals(snmpEventInfo.getPort(), def.getPort().intValue());
    	assertEquals(snmpEventInfo.getProxyHost(), def.getProxyHost());
    	assertEquals(snmpEventInfo.getRetryCount(), def.getRetry().intValue());
    	assertEquals(snmpEventInfo.getTimeout(), def.getTimeout().intValue());
    	assertEquals(snmpEventInfo.getVersion(), def.getVersion());
    	assertEquals(snmpEventInfo.getMaxRequestSize(), def.getMaxRequestSize().intValue());
    	assertEquals(snmpEventInfo.getWriteCommunityString(), def.getWriteCommunity());
    	    	
    	// v3 properties must not be set
    	assertNull(def.getAuthPassphrase());
    	assertNull(def.getAuthProtocol());
    	assertNull(def.getContextEngineId());
    	assertNull(def.getContextName());
    	assertNull(def.getEngineId());
    	assertNull(def.getEnterpriseId());
    	assertNull(def.getPrivacyPassphrase());
    	assertNull(def.getPrivacyProtocol());
    }
    
    @Test
    public void testCreateDefForSnmpV3() {
    	SnmpEventInfo snmpEventInfo = new SnmpEventInfo();
    	snmpEventInfo.setAuthPassPhrase("authPassPhrase");
    	snmpEventInfo.setAuthProtocol("authProtocol");
    	snmpEventInfo.setReadCommunityString("communityString");
    	snmpEventInfo.setWriteCommunityString("writeCommunityString");
    	snmpEventInfo.setContextEngineId("contextEngineId");
    	snmpEventInfo.setContextName("contextName");
    	snmpEventInfo.setEngineId("engineId");
    	snmpEventInfo.setEnterpriseId("enterpriseId");
    	snmpEventInfo.setMaxRepetitions(1000);
    	snmpEventInfo.setMaxVarsPerPdu(2000);
    	snmpEventInfo.setMaxRequestSize(7000);
    	snmpEventInfo.setPort(3000);
    	snmpEventInfo.setProxyHost("127.0.0.1");
    	snmpEventInfo.setPrivPassPhrase("privPassPhrase");
    	snmpEventInfo.setPrivProtocol("privProtocol");
    	snmpEventInfo.setRetryCount(4000);
    	snmpEventInfo.setSecurityLevel(5000);
    	snmpEventInfo.setTimeout(6000);
    	snmpEventInfo.setVersion("v3");
    	
    	// check v3/commons propertiess
    	Definition def = snmpEventInfo.createDef();
    	assertEquals(snmpEventInfo.getAuthPassphrase(), def.getAuthPassphrase());
    	assertEquals(snmpEventInfo.getAuthProtocol(), def.getAuthProtocol());
    	assertEquals(snmpEventInfo.getContextEngineId(), def.getContextEngineId());
    	assertEquals(snmpEventInfo.getContextName(), def.getContextName());
    	assertEquals(snmpEventInfo.getEngineId(), def.getEngineId());
    	assertEquals(snmpEventInfo.getEnterpriseId(), def.getEnterpriseId());
    	assertEquals(snmpEventInfo.getPrivPassPhrase(), def.getPrivacyPassphrase());
    	assertEquals(snmpEventInfo.getPrivProtocol(), def.getPrivacyProtocol());
    	assertEquals(snmpEventInfo.getMaxRepetitions(), def.getMaxRepetitions().intValue());
    	assertEquals(snmpEventInfo.getMaxVarsPerPdu(), def.getMaxVarsPerPdu().intValue());
    	assertEquals(snmpEventInfo.getMaxRequestSize(), def.getMaxRequestSize().intValue());
    	assertEquals(snmpEventInfo.getPort(), def.getPort().intValue());
    	assertEquals(snmpEventInfo.getProxyHost(), def.getProxyHost());
    	assertEquals(snmpEventInfo.getRetryCount(), def.getRetry().intValue());
    	assertEquals(snmpEventInfo.getTimeout(), def.getTimeout().intValue());
    	assertEquals(snmpEventInfo.getVersion(), def.getVersion());
    	
    	// v1/v2 properties must not be set
    	assertNull(def.getReadCommunity());
    	assertNull(def.getWriteCommunity());
    }
    

    /**
     * This tests the ability of a configureSNMP event to change the community
     * string of a specific address.
     */
    @Test
    public final void testModifySpecificInDef() throws Exception {
        
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "   <definition version=\"v2c\">\n" + 
        "      <specific>192.168.0.5</specific>\n" + 
        "   </definition>\n" + 
        "</snmp-config>\n" + 
        "";
        
        String expectedConfig = "" +
        "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "   <definition read-community=\"abc\">\n" + 
        "      <specific>192.168.0.5</specific>\n" + 
        "   </definition>\n" + 
        "</snmp-config>\n" +
        "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());
        
        SnmpEventInfo info = new SnmpEventInfo();
        info.setReadCommunityString("abc");
        info.setFirstIPAddress("192.168.0.5");
        
        SnmpPeerFactory.getInstance().define(info);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);

    }

    /**
     * This tests the ability of a configureSNMP event to configure v3 settings.
     */
    @Test
    public final void testConfiguringV3ThroughEventInDef() throws Exception {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "   <definition version=\"v2c\">\n" +
                "      <specific>192.168.0.5</specific>\n" +
                "   </definition>\n" +
                "</snmp-config>\n" +
                "";

        String expectedConfig = "" +
                "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "   <definition auth-passphrase=\"nant\" auth-protocol=\"SHA-256\" version=\"v3\">\n" +
                "      <specific>192.168.0.5</specific>\n" +
                "   </definition>\n" +
                "</snmp-config>\n" +
                "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));

        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        SnmpEventInfo info = new SnmpEventInfo();
        info.setFirstIPAddress("192.168.0.5");
        info.setVersion("v3");
        info.setAuthProtocol("SHA-256");
        info.setAuthPassPhrase("nant");

        SnmpPeerFactory.getInstance().define(info);

        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);

        SnmpAgentConfig agentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.0.5"));
        assertEquals(agentConfig.getVersion(), SnmpConfiguration.VERSION3);
        assertEquals(agentConfig.getAuthProtocol(), "SHA-256");
        assertEquals(agentConfig.getAuthPassPhrase(), "nant");

    }

    /**
     * This tests the ability of a configureSNMP event to change the community
     * string of a specific address.
     */
    @Test
    public final void testModifySpecificInDefIPv6() throws Exception {
        
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <specific>2001:db8::10</specific>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";
        
        String expectedConfig = "" +
        "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition read-community=\"abc\">\n" +
        "        <specific>2001:db8::10</specific>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" +
        "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());
        
        SnmpEventInfo info = new SnmpEventInfo();
        info.setReadCommunityString("abc");
        info.setFirstIPAddress("2001:db8::10");
        
        SnmpPeerFactory.getInstance().define(info);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
        
    }

    private static class StringResource extends AbstractResource {
        private final File m_tempFile;

        public StringResource(final String text) throws IOException {
            m_tempFile = File.createTempFile("StringResource", ".txt");
            FileWriter fw = null;
            try {
                fw = new FileWriter(m_tempFile);
                fw.write(text);
                fw.flush();
                fw.close();
            } finally {
                IOUtils.closeQuietly(fw);
            }
        }

        @Override
        public String getDescription() {
            return "string [" + m_tempFile.getPath() + "]";
        }

        @Override
        public InputStream getInputStream() throws IOException {
            return new FileInputStream(this.m_tempFile);
        }
    }

    /**
     * This tests the merging of a new specific into a definition that already contains a specific
     * that is adjacent.  The two specifics should be converted to a single range in the definition.
     */
    @Test
    public final void testAddAdjacentSpecificToDef() throws Exception {

        String snmpConfigXml = "" +
        		"<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        		"<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        		"    <definition version=\"v2c\">\n" + 
        		"        <specific>192.168.0.5</specific>\n" + 
        		"    </definition>\n" + 
        		"</snmp-config>\n" +
        		"";

        String expectedConfig = "" +
        "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.0.5\" end=\"192.168.0.6\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" +
        "";


        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());
        
        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v2c");
        info.setFirstIPAddress("192.168.0.6");
        
        SnmpPeerFactory.getInstance().define(info);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
        
    }


    /**
     * This tests the merging of a new specific into a definition that already contains a specific
     * that is adjacent.  The two specifics should be converted to a single range in the definition.
     * 
     */
    @Test
    public final void testAddAdjacentSpecificToDefIPv6() throws Exception {

        String snmpConfigXml = "" +
                "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
                "    <definition version=\"v2c\">\n" + 
                "        <specific>2001:db8::10</specific>\n" + 
                "    </definition>\n" + 
                "</snmp-config>\n" +
                "";

        String expectedConfig = "" +
        "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"2001:db8::10\" end=\"2001:db8::11\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" +
        "";


        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());
        
        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v2c");
        info.setFirstIPAddress("2001:db8::11");
        
        SnmpPeerFactory.getInstance().define(info);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
        
    }
    /**
     * This tests the merging of a new specific into a definition that already contains a specific
     * that is adjacent.  The two specifics should be converted to a single range in the definition.
     */
    @Test
    public final void testAddSpecificBetweenAdjacentsSpecifics() throws Exception {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <specific>192.168.0.5</specific>\n" + 
        "        <specific>192.168.0.7</specific>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.0.5\" end=\"192.168.0.7\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());
        
        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v2c");
        info.setFirstIPAddress("192.168.0.6");
        
        SnmpPeerFactory.getInstance().define(info);

        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
        
    }

    /**
     * This tests the merging of a new specific into a definition that already contains a specific
     * that is adjacent.  The two specifics should be converted to a single range in the definition.
     */
    @Test
    public final void testAddSpecificBetweenAdjacentsSpecificsMostlyZeros() throws Exception {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <specific>0.0.0.5</specific>\n" + 
        "        <specific>0.0.0.7</specific>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"0.0.0.5\" end=\"0.0.0.7\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());
        
        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v2c");
        info.setFirstIPAddress("0.0.0.6");
        
        SnmpPeerFactory.getInstance().define(info);

        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
        
    }

    /**
     * This tests the merging of a new specific into a definition that already contains a specific
     * that is adjacent.  The two specifics should be converted to a single range in the definition.
     */
    @Test
    public final void testAddSpecificBetweenAdjacentsSpecificsIPv6() throws Exception {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <specific>::5</specific>\n" + 
        "        <specific>::7</specific>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"::5\" end=\"::7\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());
        
        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v2c");
        info.setFirstIPAddress("::6");
        
        SnmpPeerFactory.getInstance().define(info);

        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
        
    }
    @Test
    public final void testAddSpecificBetweenAdjacentSpecificAndRange() throws Exception {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.0.7\" end=\"192.168.0.9\"/>\n" + 
        "        <specific>192.168.0.5</specific>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.0.5\" end=\"192.168.0.9\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());
        
        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v2c");
        info.setFirstIPAddress("192.168.0.6");
        
        SnmpPeerFactory.getInstance().define(info);

        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
        
}
    /**
     * This tests the ability to add a new range of IPs that create a new definition and these IPs
     * were previously defined withing the a range of a current definition.  The result should be
     * that the previous range is split into 2 ranges and the new range is now contained in a new
     * separate definition.
     */
    @Test
    public void testSplitRange() throws Exception {
        
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.100\" end=\"192.168.1.200\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";
        
        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.100\" end=\"192.168.1.119\"/>\n" + 
        "        <range begin=\"192.168.1.131\" end=\"192.168.1.200\"/>\n" + 
        "    </definition>\n" + 
        "    <definition version=\"v1\">\n" + 
        "        <range begin=\"192.168.1.120\" end=\"192.168.1.130\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";
        
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());


        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v1");
        info.setFirstIPAddress("192.168.1.120");
        info.setLastIPAddress("192.168.1.130");

        SnmpPeerFactory.getInstance().define(info);
        
//        String config = SnmpPeerFactory.marshallConfig();
//        System.err.println(config);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
        
    }

    /**
     * This tests the ability to create a new definition from an IP that was previously covered by
     * a range in an existing definition.  The result should be the previous range is converted into 2
     * new ranges and the new specific is added to a new definition in the config.
     */
    @Test
    public void testRemoveSpecificFromRange() throws Exception {
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.100\" end=\"192.168.1.200\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.100\" end=\"192.168.1.119\"/>\n" + 
        "        <range begin=\"192.168.1.121\" end=\"192.168.1.200\"/>\n" + 
        "    </definition>\n" + 
        "    <definition version=\"v1\">\n" + 
        "        <specific>192.168.1.120</specific>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());


        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v1");
        info.setFirstIPAddress("192.168.1.120");

        SnmpPeerFactory.getInstance().define(info);
        
//        String config = SnmpPeerFactory.marshallConfig();
//        System.err.println(config);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
        
    }
    
    @Test
    public void testRemoveSpecificNearEndOfRange() throws Exception {
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.100\" end=\"192.168.1.200\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.100\" end=\"192.168.1.198\"/>\n" + 
        "        <specific>192.168.1.200</specific>\n" + 
        "    </definition>\n" + 
        "    <definition version=\"v1\">\n" + 
        "        <specific>192.168.1.199</specific>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());


        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v1");
        info.setFirstIPAddress("192.168.1.199");

        SnmpPeerFactory.getInstance().define(info);
        
//        String config = SnmpPeerFactory.marshallConfig();
//        System.err.println(config);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
        
    }

    @Test
    public void testRemoveSpecificAtEndOfRange() throws Exception {
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.100\" end=\"192.168.1.200\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.100\" end=\"192.168.1.199\"/>\n" + 
        "    </definition>\n" + 
        "    <definition version=\"v1\">\n" + 
        "        <specific>192.168.1.200</specific>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        SnmpPeerFactory factory = new SnmpPeerFactory(new StringResource(snmpConfigXml));
        SnmpPeerFactory.setInstance(factory);
        
        assertXmlEquals(snmpConfigXml, factory.getSnmpConfigAsString());


        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v1");
        info.setFirstIPAddress("192.168.1.200");

        SnmpPeerFactory.getInstance().define(info);
        
//        String config = SnmpPeerFactory.marshallConfig();
//        System.err.println(config);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
        
    }
    /**
     * This tests the ability to move a specific from one definition into a range of another definition.  The
     * results should be that the 2 ranges in the first definition are recombined into a single range based on 
     * the single IP address that was in a different existing defintion that will now be removed and the definition
     * deleted.
     */
    @Test
    public void testRecombineSpecificIntoRange() throws Exception {
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.10\" end=\"192.168.1.14\"/>\n" + 
        "        <range begin=\"192.168.1.16\" end=\"192.168.1.40\"/>\n" + 
        "    </definition>\n" + 
        "    <definition version=\"v1\">\n" + 
        "        <specific>192.168.1.15</specific>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";
        
        String expectedConfig = 
        "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.10\" end=\"192.168.1.40\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";    

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());


        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v2c");
        info.setFirstIPAddress("192.168.1.15");

        SnmpPeerFactory.getInstance().define(info);
        
//        String config = SnmpPeerFactory.marshallConfig();
//        System.err.println(config);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
        
    }
    
    /**
     * This tests the ability to remove a specific IP from one definition with a newly specified range.
     */
    @Test
    public void testRemoveSpecificInSeparateDefWithNewRange() throws Exception {
        
        String snmpConfigXml = 
            "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        		"<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        		"    <definition version=\"v2c\">\n" + 
        		"        <specific>192.168.1.30</specific>\n" + 
        		"        <specific>10.1.1.1</specific>\n" + 
        		"    </definition>\n" + 
        		"</snmp-config>\n" + 
        		"";

        String expectedConfig = 
        "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <specific>10.1.1.1</specific>\n" + 
        "    </definition>\n" + 
        "    <definition version=\"v1\">\n" + 
        "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());


        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v1");
        info.setFirstIPAddress("192.168.1.15");
        info.setLastIPAddress("192.168.1.35");

        SnmpPeerFactory.getInstance().define(info);
        
//        String config = SnmpPeerFactory.marshallConfig();
//        System.err.println(config);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);

    }
    
    @Test
    public void testRemoveTrivialEntry() throws Exception {
        
        String snmpConfigXml = 
            "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
                "    <definition version=\"v2c\">\n" + 
                "        <specific>192.168.1.30</specific>\n" + 
                "        <specific>10.1.1.1</specific>\n" + 
                "    </definition>\n" + 
                "</snmp-config>\n" + 
                "";

        String expectedConfig = 
        "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <specific>10.1.1.1</specific>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());


        SnmpEventInfo info = new SnmpEventInfo();
        info.setFirstIPAddress("192.168.1.30");

        SnmpPeerFactory.getInstance().define(info);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);

    }

    /**
     * This tests the behavior of specifying an invalid range.
     * @throws IOException 
     */
    @Test
    public void testInvalidRange() throws IOException {
        String snmpConfigXml = "<?xml version=\"1.0\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "   <definition version=\"v2c\">\n" + 
        "       <range begin=\"192.168.0.3\" end=\"192.168.0.100\"/>" + 
        "   </definition>\n" + 
        "\n" + 
        "</snmp-config>\n" + 
        "";
                
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));

        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v1");
        info.setFirstIPAddress("192.168.0.3");
        info.setLastIPAddress("192.168.0.1");

        try {
            SnmpPeerFactory.getInstance().define(info);
            fail("Expected IllegalArgumentException.");
        } catch (IllegalArgumentException e) {
            
        }
    }
    
    /**
     * This tests the addition of a new specific definition that is the same address as the beginning of
     * a range in a current definition.
     */
    @Test
    public void testNewSpecifcSameAsBeginInOldDef() throws Exception {
        
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.0.3\" end=\"192.168.0.100\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";
        
        String expectedConfig = 
        "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.0.4\" end=\"192.168.0.100\"/>\n" + 
        "    </definition>\n" + 
        "    <definition version=\"v1\">\n" + 
        "        <specific>192.168.0.3</specific>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";
        

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());


        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v1");
        info.setFirstIPAddress("192.168.0.3");

        SnmpPeerFactory.getInstance().define(info);
        
//        String config = SnmpPeerFactory.marshallConfig();
//        System.err.println(config);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);

    }
    
    /**
     * This tests the merging of a new definition that contains a range of IP addresses that overlaps
     * the end of one range and the beginning of another range in a current definition.
     */
    @Test
    public void testOverlapsTwoRanges() throws Exception {
        
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        		"<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        		"    <definition version=\"v2c\">\n" + 
        		"        <range begin=\"192.168.1.10\" end=\"192.168.1.20\"/>\n" + 
        		"        <range begin=\"192.168.1.30\" end=\"192.168.1.40\"/>\n" + 
        		"    </definition>\n" + 
        		"</snmp-config>\n" + 
        		"";
        
        String expectedConfig = 
        "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.10\" end=\"192.168.1.14\"/>\n" + 
        "        <range begin=\"192.168.1.36\" end=\"192.168.1.40\"/>\n" + 
        "    </definition>\n" + 
        "    <definition version=\"v1\">\n" + 
        "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());


        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v1");
        info.setFirstIPAddress("192.168.1.15");
        info.setLastIPAddress("192.168.1.35");

        SnmpPeerFactory.getInstance().define(info);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
    }

    public void testOverlapsTwoRangesAndCombinesThem(String firstIp, String lastIp) throws Exception {
        
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
                "    <definition version=\"v2c\">\n" + 
                "        <range begin=\"192.168.1.10\" end=\"192.168.1.20\"/>\n" + 
                "        <range begin=\"192.168.1.30\" end=\"192.168.1.40\"/>\n" + 
                "    </definition>\n" + 
                "</snmp-config>\n" + 
                "";
        
        String expectedConfig = 
        "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.10\" end=\"192.168.1.40\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());


        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v2c");
        info.setFirstIPAddress(firstIp);
        info.setLastIPAddress(lastIp);

        SnmpPeerFactory.getInstance().define(info);

        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
    }
    
    public void testOverlapsTwoRangesWithLocationAndCombinesThem(String firstIp, String lastIp) throws Exception {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n"
                + "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n"
                + "    <definition location=\"MINION\" version=\"v2c\">\n"
                + "        <range begin=\"192.168.1.10\" end=\"192.168.1.20\"/>\n"
                + "        <range begin=\"192.168.1.30\" end=\"192.168.1.40\"/>\n" + "    </definition>\n"
                + "    <definition location=\"AUSTIN\" version=\"v2c\">\n"
                + "        <range begin=\"192.168.2.10\" end=\"192.168.2.20\"/>\n"
                + "        <range begin=\"192.168.2.30\" end=\"192.168.2.40\"/>\n" + "    </definition>\n"
                + "</snmp-config>\n" + "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n"
                + "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n"
                + "    <definition location=\"MINION\" version=\"v2c\">\n"
                + "        <range begin=\"192.168.1.10\" end=\"192.168.1.20\"/>\n"
                + "        <range begin=\"192.168.1.30\" end=\"192.168.1.40\"/>\n" + "    </definition>\n"
                + "    <definition location=\"AUSTIN\" version=\"v2c\">\n"
                + "        <range begin=\"192.168.1.10\" end=\"192.168.1.40\"/>\n"
                + "        <range begin=\"192.168.2.10\" end=\"192.168.2.20\"/>\n"
                + "        <range begin=\"192.168.2.30\" end=\"192.168.2.40\"/>\n" + "    </definition>\n"
                + "</snmp-config>\n" + "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));

        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v2c");
        info.setLocation("AUSTIN");
        info.setFirstIPAddress(firstIp);
        info.setLastIPAddress(lastIp);

        SnmpPeerFactory.getInstance().define(info);

        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
    }

    @Test
    public void testCombineOverlappingRanges() throws Exception {
        testOverlapsTwoRangesAndCombinesThem("192.168.1.15", "192.168.1.35");
        testOverlapsTwoRangesAndCombinesThem("192.168.1.10", "192.168.1.35");
        testOverlapsTwoRangesAndCombinesThem("192.168.1.20", "192.168.1.35");
        testOverlapsTwoRangesAndCombinesThem("192.168.1.21", "192.168.1.35");
        testOverlapsTwoRangesAndCombinesThem("192.168.1.15", "192.168.1.40");
        testOverlapsTwoRangesAndCombinesThem("192.168.1.21", "192.168.1.30");
        testOverlapsTwoRangesAndCombinesThem("192.168.1.21", "192.168.1.29");
        testOverlapsTwoRangesAndCombinesThem("192.168.1.10", "192.168.1.40");

    }

    @Test
    public void testCombineOverLappingRangesWithLocation() throws Exception {
        testOverlapsTwoRangesWithLocationAndCombinesThem("192.168.1.10", "192.168.1.40");
        testOverlapsTwoRangesWithDefaultLocationAndCombinesThem("192.168.1.0", "192.168.1.40");
        testOverlapsTwoRangesWithoutLocationAndCombinesThem("192.168.1.15", "192.168.1.40");
    }

    private void testOverlapsTwoRangesWithDefaultLocationAndCombinesThem(String firstIp, String lastIp)
            throws IOException {
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n"
                + "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n"
                + "    <definition location=\"MINION\" version=\"v2c\">\n"
                + "        <range begin=\"192.168.1.10\" end=\"192.168.1.20\"/>\n"
                + "        <range begin=\"192.168.1.30\" end=\"192.168.1.40\"/>\n" + "    </definition>\n"
                + "    <definition location=\"Default\" version=\"v2c\">\n"
                + "        <range begin=\"192.168.2.10\" end=\"192.168.2.20\"/>\n"
                + "        <range begin=\"192.168.2.30\" end=\"192.168.2.40\"/>\n" + "    </definition>\n"
                + "</snmp-config>\n" + "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n"
                + "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n"
                + "    <definition location=\"MINION\" version=\"v2c\">\n"
                + "        <range begin=\"192.168.1.10\" end=\"192.168.1.20\"/>\n"
                + "        <range begin=\"192.168.1.30\" end=\"192.168.1.40\"/>\n" + "    </definition>\n"
                + "    <definition version=\"v2c\">\n"
                + "        <range begin=\"192.168.1.0\" end=\"192.168.1.40\"/>\n"
                + "        <range begin=\"192.168.2.10\" end=\"192.168.2.20\"/>\n"
                + "        <range begin=\"192.168.2.30\" end=\"192.168.2.40\"/>\n" + "    </definition>\n"
                + "</snmp-config>\n" + "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));

        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v2c");
        info.setFirstIPAddress(firstIp);
        info.setLastIPAddress(lastIp);

        SnmpPeerFactory.getInstance().define(info);

        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);

    }

    private void testOverlapsTwoRangesWithoutLocationAndCombinesThem(String firstIp, String lastIp) throws IOException {
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n"
                + "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n"
                + "    <definition location=\"MINION\" version=\"v2c\">\n"
                + "        <range begin=\"192.168.1.10\" end=\"192.168.1.20\"/>\n"
                + "        <range begin=\"192.168.1.30\" end=\"192.168.1.40\"/>\n" + "    </definition>\n"
                + "    <definition version=\"v2c\">\n"
                + "        <range begin=\"192.168.2.10\" end=\"192.168.2.20\"/>\n"
                + "        <range begin=\"192.168.2.30\" end=\"192.168.2.40\"/>\n" + "    </definition>\n"
                + "</snmp-config>\n" + "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n"
                + "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n"
                + "    <definition location=\"MINION\" version=\"v2c\">\n"
                + "        <range begin=\"192.168.1.10\" end=\"192.168.1.20\"/>\n"
                + "        <range begin=\"192.168.1.30\" end=\"192.168.1.40\"/>\n" + "    </definition>\n"
                + "    <definition version=\"v2c\">\n"
                + "        <range begin=\"192.168.1.15\" end=\"192.168.1.40\"/>\n"
                + "        <range begin=\"192.168.2.10\" end=\"192.168.2.20\"/>\n"
                + "        <range begin=\"192.168.2.30\" end=\"192.168.2.40\"/>\n" + "    </definition>\n"
                + "</snmp-config>\n" + "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));

        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v2c");
        info.setFirstIPAddress(firstIp);
        info.setLastIPAddress(lastIp);
        info.setLocation("Default");

        SnmpPeerFactory.getInstance().define(info);

        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);

    }

    /**
     * This tests moving a range that from one defintion into another defintion for which it overlaps
     * one range in the merging definition and creates 2 adjacent ranges that should be merged together.
     */
    @Test
    public void testRecombineRanges() throws Exception {
        
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.10\" end=\"192.168.1.14\"/>\n" + 
        "        <range begin=\"192.168.1.36\" end=\"192.168.1.40\"/>\n" + 
        "    </definition>\n" + 
        "    <definition version=\"v1\">\n" + 
        "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";
        
        String expectedConfig = 
        "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.10\" end=\"192.168.1.40\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";            
        
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());


        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v2c");
        info.setFirstIPAddress("192.168.1.15");
        info.setLastIPAddress("192.168.1.35");

        SnmpPeerFactory.getInstance().define(info);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);

    }
    
    /**
     * This tests recombing a range into an existing defintion for which the new range overlaps the current definition's 
     * range and overlaps the end of one range and the beginning of another range of the merging definition.
     */
    @Test
    public void testRecombineRangesNonAdjacentRange() throws Exception {
        
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.10\" end=\"192.168.1.14\"/>\n" + 
        "        <range begin=\"192.168.1.36\" end=\"192.168.1.40\"/>\n" + 
        "    </definition>\n" + 
        "    <definition version=\"v1\">\n" + 
        "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";
        
        String expectedConfig = 
        "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <range begin=\"192.168.1.10\" end=\"192.168.1.40\"/>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());


        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v2c");
        info.setFirstIPAddress("192.168.1.12");
        info.setLastIPAddress("192.168.1.38");

        SnmpPeerFactory.getInstance().define(info);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
    }

    /**
     * This tests moving the specific from a current defition to a new defintion.
     */
    @Test
    public final void testAddNewSpecificToConfig() throws Exception {
        
        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <specific>192.168.0.5</specific>\n" + 
        "        <specific>192.168.0.6</specific>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";
        
        String expectedConfig =
        "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
        "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" + 
        "    <definition version=\"v2c\">\n" + 
        "        <specific>192.168.0.5</specific>\n" + 
        "    </definition>\n" + 
        "    <definition version=\"v1\">\n" + 
        "        <specific>192.168.0.6</specific>\n" + 
        "    </definition>\n" + 
        "</snmp-config>\n" + 
        "";

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        
        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v1");
        info.setFirstIPAddress("192.168.0.6");
        
        SnmpPeerFactory.getInstance().define(info);

        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);


    }
    
    /**
     * Tests that the new definition set by the SnmpEventInfo matches the defaults.
     * Therefore a new defintion should NOT BE added.
     * @throws Exception 
     */
    @Test
    public void testEmptySnmpConfigAddDefinitionWhichMatchesDefaults() throws Exception {
    	final String snmpConfigXml = 
		"<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
		"<snmp-config port=\"161\" retry=\"3\" timeout=\"800\" read-community=\"public\" version=\"v2c\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\"/>\n";
    	final String expectedConfig = 
		"<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
		"<snmp-config port=\"161\" retry=\"3\" timeout=\"800\" read-community=\"public\" version=\"v2c\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\"/>\n";
    	
    	  SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
          assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

          SnmpEventInfo info = new SnmpEventInfo();
          info.setVersion("v2c");
          info.setTimeout(800);
          info.setFirstIPAddress("192.168.0.8");
          
          SnmpPeerFactory.getInstance().define(info);
          
          String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
          assertXmlEquals(expectedConfig, actualConfig);
    }


    /**
     * Tests removal of IP address from definition.
     * @throws IOException
     */
    @Test
    public void testRemovalOfIpAddressFromRangeFromDefinition() throws IOException {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config port=\"161\" retry=\"3\" timeout=\"800\" read-community=\"public\" version=\"v1\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v2c\" location=\"MINION\" port=\"1161\">\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config port=\"161\" retry=\"3\" timeout=\"800\" read-community=\"public\" version=\"v1\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v2c\" location=\"MINION\" port=\"1161\">\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.34\"/>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        boolean success = SnmpPeerFactory.getInstance().removeFromDefinition(InetAddress.getByName("192.168.1.35"), "MINION","test" );
        assertTrue("Remove from definition should be successful", success);
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        System.out.println(actualConfig);
        assertXmlEquals(expectedConfig, actualConfig);
    }


    /**
     * Tests removal of IP address from middle of range splits the range.
     * @throws IOException
     */
    @Test
    public void testRemovalOfIpAddressRangeShouldSplit() throws IOException {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\" timeout=\"3000\" write-community=\"private\"  >\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\" timeout=\"3000\" write-community=\"private\"  >\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.24\"/>\n" +
                "        <range begin=\"192.168.1.26\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        boolean success = SnmpPeerFactory.getInstance().removeFromDefinition(InetAddress.getByName("192.168.1.25"), null, "test");
        assertTrue("Remove from definition should be successful", success);
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
    }

    /**
     * Tests removal of IP address with location that is not matching definition wouldn't change definitions.
     * @throws IOException
     */
    @Test
    public void testRemovalOfIpAddressWithDifferentLocation() throws IOException {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\">\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\">\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        boolean success = SnmpPeerFactory.getInstance().removeFromDefinition(InetAddress.getByName("192.168.1.25"), "MINION", "test");
        assertFalse("Remove from definition should fail", success);
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
    }

    /**
     * Tests removal of IP address from definition with matching location.
     * @throws IOException
     */
    @Test
    public void testRemovalOfIpAddressWithDifferentLocationsInDefinitions() throws IOException {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\"  timeout=\"1200\" write-community=\"horizon\" >\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "    <definition version=\"v1\" write-community=\"horizon\" location=\"Minion\"> \n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\"  xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\" timeout=\"1200\" write-community=\"horizon\" >\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "    <definition version=\"v1\" write-community=\"horizon\" location=\"Minion\"> \n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.24\"/>\n" +
                "        <range begin=\"192.168.1.26\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());
        boolean success = SnmpPeerFactory.getInstance().removeFromDefinition(InetAddress.getByName("192.168.1.25"), "Minion", "test");
        assertTrue("Remove from definition should be successful", success);
        SnmpAgentConfig snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.1.25"), "Minion");
        assertFalse("Config should be derived from default location", snmpAgentConfig.isDefault());
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();

        assertXmlEquals(expectedConfig, actualConfig);
    }


    /**
     * Tests removal of one IP address which has only one definition with one specific.
     * @throws IOException
     */
    @Test
    public void testRemovalOfIpAddressShouldDeleteDefinitions() throws IOException {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\">\n" +
                "        <specific>192.168.0.8</specific>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "</snmp-config>\n" +
                "";
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        boolean success = SnmpPeerFactory.getInstance().removeFromDefinition(InetAddress.getByName("192.168.0.8"), null, "test");
        assertTrue("Remove from definition should be successful", success);
        SnmpAgentConfig snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.0.8"));
        assertTrue("Should fall back to default config", snmpAgentConfig.isDefault());
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();

        assertXmlEquals(expectedConfig, actualConfig);
    }

    /**
     * Tests removal of one IP address which has only one definition with one specific and definition has profile label
     * @throws IOException
     */
    @Test
    public void testRemovalOfIpAddressWithProfileShouldDeleteDefinitions() throws IOException {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\" profile-label=\"profile3\">\n" +
                "        <specific>192.168.0.8</specific>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "</snmp-config>\n" +
                "";
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        boolean success = SnmpPeerFactory.getInstance().removeFromDefinition(InetAddress.getByName("192.168.0.8"), null, "test");
        assertTrue("Remove from definition should be successful", success);
        SnmpAgentConfig snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.0.8"));
        assertTrue("Should fall back to default config", snmpAgentConfig.isDefault());
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();

        assertXmlEquals(expectedConfig, actualConfig);
    }

    /**
     * Tests removal of one IP address which has only one definition with one specific and definition has profile label and location set.
     * @throws IOException
     */
    @Test
    public void testRemovalOfIpAddressWithProfileAndLocationShouldDeleteDefinitions() throws IOException {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\" profile-label=\"profile3\"  location=\"MINION\">\n" +
                "        <specific>192.168.0.8</specific>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "</snmp-config>\n" +
                "";
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        boolean success = SnmpPeerFactory.getInstance().removeFromDefinition(InetAddress.getByName("192.168.0.8"), "MINION", "test");
        assertTrue("Remove from definition should be successful", success);
        SnmpAgentConfig snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.0.8"), "MINION");
        assertTrue("Should fall back to default config", snmpAgentConfig.isDefault());
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();

        assertXmlEquals(expectedConfig, actualConfig);
    }

    /**
     * Tests removal of one IP address which has only one definition with range and definition has profile label and location set.
     * @throws IOException
     */
    @Test
    public void testRemovalOfIpAddressWithProfileAndLocationShouldModifyRange() throws IOException {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\" timeout=\"1200\" profile-label=\"profile3\" location=\"MINION\" >\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\" timeout=\"1200\" profile-label=\"profile3\" location=\"MINION\" >\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.24\"/>\n" +
                "        <range begin=\"192.168.1.26\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        boolean success = SnmpPeerFactory.getInstance().removeFromDefinition(InetAddress.getByName("192.168.1.25"), "MINION", "test");
        assertTrue("Remove from definition should be successful", success);
        SnmpAgentConfig snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.1.25"), "MINION");
        assertTrue("Should fall back to default config", snmpAgentConfig.isDefault());
        assertNull("Profile label should be null", snmpAgentConfig.getProfileLabel());
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();

        assertXmlEquals(expectedConfig, actualConfig);
    }

    /**
     * Tests removal of one IP address with multiple definitions with profile labels and location set.
     * @throws IOException
     */
    @Test
    public void testRemovalOfIpAddressWithMultipleProfilesShouldModifyRange() throws IOException {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\" timeout=\"1200\" profile-label=\"profile3\" location=\"MINION\" >\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "    <definition version=\"v1\" timeout=\"1500\" profile-label=\"profile2\" location=\"MINION\" >\n" +
                "        <range begin=\"192.168.2.15\" end=\"192.168.2.35\"/>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\" timeout=\"1200\" profile-label=\"profile3\" location=\"MINION\" >\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.24\"/>\n" +
                "        <range begin=\"192.168.1.26\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "    <definition version=\"v1\" timeout=\"1500\" profile-label=\"profile2\" location=\"MINION\" >\n" +
                "        <range begin=\"192.168.2.15\" end=\"192.168.2.35\"/>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        // Should fetch config from the definition with profile3
        SnmpAgentConfig snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.1.25"), "MINION");
        assertEquals("profile3", snmpAgentConfig.getProfileLabel());
        assertFalse("Should not be default config", snmpAgentConfig.isDefault());
        assertEquals(1200, snmpAgentConfig.getTimeout());

        // Deletion of the IP Address from definitions should fall back to default
        boolean success = SnmpPeerFactory.getInstance().removeFromDefinition(InetAddress.getByName("192.168.1.25"), "MINION", "test");
        assertTrue("Remove from definition should be successful", success);
        snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.1.25"), "MINION");
        assertTrue("Should fall back to default config", snmpAgentConfig.isDefault());
        assertNull("Profile label shouldn't be set for default config", snmpAgentConfig.getProfileLabel());
        assertEquals(800, snmpAgentConfig.getTimeout());

        snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.2.25"), "MINION");
        assertEquals("profile2", snmpAgentConfig.getProfileLabel());
        assertFalse("Should not be default config", snmpAgentConfig.isDefault());
        assertEquals(1500, snmpAgentConfig.getTimeout());

        snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.1.35"), "MINION");
        assertEquals("profile3", snmpAgentConfig.getProfileLabel());
        assertFalse("Should not be default config", snmpAgentConfig.isDefault());
        assertEquals(1200, snmpAgentConfig.getTimeout());

        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
    }

    /**
     * Tests removal of an IP Address which has different definition at a location and with default location
     * @throws IOException
     */
    @Test
    public void testRemovalOfIpAddressWithMultipleProfilesWithDifferentLocations() throws IOException {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\" timeout=\"1200\" profile-label=\"profile2\" location=\"MINION\" >\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "    <definition version=\"v1\" timeout=\"1500\" profile-label=\"profile3\" >\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "    <definition version=\"v1\" timeout=\"2000\" profile-label=\"profile4\" location=\"Apex\" >\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\" timeout=\"1200\" profile-label=\"profile2\" location=\"MINION\" >\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.24\"/>\n" +
                "        <range begin=\"192.168.1.26\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "    <definition version=\"v1\" timeout=\"1500\" profile-label=\"profile3\" >\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "    <definition version=\"v1\" timeout=\"2000\" profile-label=\"profile4\" location=\"Apex\" >\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "</snmp-config>\n" +
                "";
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        // Should fetch config from the definition with profile3
        SnmpAgentConfig snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.1.25"), "MINION");
        assertEquals("profile2", snmpAgentConfig.getProfileLabel());
        assertFalse("Should not be default config", snmpAgentConfig.isDefault());
        assertEquals(1200, snmpAgentConfig.getTimeout());

        // Deletion of the IP Address from definitions should fall back to default
        boolean success = SnmpPeerFactory.getInstance().removeFromDefinition(InetAddress.getByName("192.168.1.25"), "MINION", "test");
        assertTrue("Remove from definition should be successful", success);
        snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.1.25"), "MINION");
        assertFalse("Should fall back to config from default location", snmpAgentConfig.isDefault());
        assertEquals("profile3", snmpAgentConfig.getProfileLabel());
        assertEquals(1500, snmpAgentConfig.getTimeout());


        snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.1.35"), "MINION");
        assertEquals("profile2", snmpAgentConfig.getProfileLabel());
        assertFalse("Should not be default config", snmpAgentConfig.isDefault());
        assertEquals(1200, snmpAgentConfig.getTimeout());

        snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.1.35"), "Apex");
        assertEquals("profile4", snmpAgentConfig.getProfileLabel());
        assertFalse("Should not be default config", snmpAgentConfig.isDefault());
        assertEquals(2000, snmpAgentConfig.getTimeout());

        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
    }





    /**
     * Tests that a new snmp config from profile creates a new definition.
     * @throws IOException
     */
    @Test
    public void testSnmpProfilesWithDifferentConfiguration() throws IOException {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\">\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "       <profiles>" +
                "           <profile " + "version=\"v2c\">" +
                "             <label>sample</label>" +
                "           </profile>\n" +
                "       </profiles>\n" +
                "</snmp-config>\n" +
                "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\">\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.24\"/>\n" +
                "        <range begin=\"192.168.1.26\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "    <definition version=\"v2c\" profile-label=\"sample\">\n" +
                "        <specific>192.168.1.25</specific>\n" +
                "    </definition>\n" +
                "       <profiles>" +
                "           <profile " + "version=\"v2c\">" +
                "             <label>sample</label>" +
                "           </profile>\n" +
                "       </profiles>\n" +
                "</snmp-config>\n" +
                "";
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        SnmpProfile snmpProfile = SnmpPeerFactory.getInstance().getProfiles().get(0);
        SnmpAgentConfig snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfigFromProfile(snmpProfile, InetAddress.getByName("192.168.1.25"));
        assertFalse("Config should be not be default config", snmpAgentConfig.isDefault());
        SnmpPeerFactory.getInstance().saveAgentConfigAsDefinition(snmpAgentConfig, null, "test");
        snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfigFromProfile(snmpProfile, InetAddress.getByName("192.168.1.25"));
        assertFalse("Config should be not be default config", snmpAgentConfig.isDefault());
        assertEquals("sample", snmpAgentConfig.getProfileLabel());
        snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfig(InetAddress.getByName("192.168.1.24"));
        assertFalse("Config should be not be default config", snmpAgentConfig.isDefault());
        assertNull("Profile label should be null", snmpAgentConfig.getProfileLabel());
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
    }


    /**
     * Tests that a profile matching config with config definition merges the ipaddress with range.
     * @throws IOException
     */
    @Test
    public void testSnmpProfilesWithConfigMatchingDefinition() throws IOException {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\" profile-label=\"sample\">\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.24\"/>\n" +
                "        <range begin=\"192.168.1.26\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "       <profiles>" +
                "           <profile " + "version=\"v1\">" +
                "             <label>sample</label>" +
                "           </profile>\n" +
                "       </profiles>\n" +
                "</snmp-config>\n" +
                "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\" profile-label=\"sample\">\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "       <profiles>" +
                "           <profile " + "version=\"v1\">" +
                "             <label>sample</label>" +
                "           </profile>\n" +
                "       </profiles>\n" +
                "</snmp-config>\n" +
                "";
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        SnmpProfile snmpProfile = SnmpPeerFactory.getInstance().getProfiles().get(0);
        SnmpAgentConfig snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfigFromProfile(snmpProfile, InetAddress.getByName("192.168.1.25"));
        assertFalse("Config should be not be default config", snmpAgentConfig.isDefault());
        assertEquals("sample", snmpAgentConfig.getProfileLabel());
        SnmpPeerFactory.getInstance().saveAgentConfigAsDefinition(snmpAgentConfig, null, "test");
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
    }

    /**
     * Tests that a definition without profile label won't merge with other definition even if config is same.
     * @throws IOException
     */
    @Test
    public void testSnmpProfilesWithProfileLabelInDefinition() throws IOException {

        String snmpConfigXml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\">\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.24\"/>\n" +
                "        <range begin=\"192.168.1.26\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "       <profiles>" +
                "           <profile " + "version=\"v1\">" +
                "             <label>sample</label>" +
                "           </profile>\n" +
                "       </profiles>\n" +
                "</snmp-config>\n" +
                "";

        String expectedConfig = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
                "<snmp-config retry=\"3\" timeout=\"800\" read-community=\"public\" write-community=\"private\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n" +
                "    <definition version=\"v1\">\n" +
                "        <range begin=\"192.168.1.15\" end=\"192.168.1.24\"/>\n" +
                "        <range begin=\"192.168.1.26\" end=\"192.168.1.35\"/>\n" +
                "    </definition>\n" +
                "    <definition version=\"v1\" profile-label=\"sample\">\n" +
                "         <specific>192.168.1.25</specific>\n" +
                "    </definition>\n" +
                "       <profiles>" +
                "           <profile " + "version=\"v1\">" +
                "             <label>sample</label>" +
                "           </profile>\n" +
                "       </profiles>\n" +
                "</snmp-config>\n" +
                "";
        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        SnmpProfile snmpProfile = SnmpPeerFactory.getInstance().getProfiles().get(0);
        SnmpAgentConfig snmpAgentConfig = SnmpPeerFactory.getInstance().getAgentConfigFromProfile(snmpProfile, InetAddress.getByName("192.168.1.25"));
        assertFalse("Config should be not be default config", snmpAgentConfig.isDefault());
        SnmpPeerFactory.getInstance().saveAgentConfigAsDefinition(snmpAgentConfig, null, "test");
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
    }




    /**
     * In earlier Versions of OpenNMS max-repetitions and max-vars-per-pdu weren't considered in the optimization.
     * So this test checks if it is now considered.
     */
    @Test
    public void testMaxRepetitionsAndMaxVarsPerPdu() throws Exception {
    	final String snmpConfigXml = 
		"<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
		"<snmp-config port=\"161\" retry=\"3\" timeout=\"800\" read-community=\"public\" version=\"v1\" max-repetitions=\"17\" max-vars-per-pdu=\"13\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\"/>\n";
    	
    	final String expectedConfig = 
		"<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
		"<snmp-config port=\"161\" retry=\"3\" timeout=\"800\" read-community=\"public\" version=\"v1\" max-repetitions=\"17\" max-vars-per-pdu=\"13\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n"+
		"    <definition version=\"v2c\" max-repetitions=\"5\">\n" + 
		"        <specific>192.168.0.8</specific>\n" + 
		"    </definition>\n" + 
		"</snmp-config>\n";
    	

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        SnmpEventInfo info = new SnmpEventInfo();
        info.setVersion("v2c");
        info.setMaxVarsPerPdu(13);
        info.setMaxRepetitions(5);
        info.setFirstIPAddress("192.168.0.8");
        
        SnmpPeerFactory.getInstance().define(info);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
    }
    
    /**
     * Tests if the proxy host is considered in the optimization code.
     */
    @Test
    public void testProxyHost() throws Exception {
    	final String snmpConfigXml = 
		"<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
		"<snmp-config port=\"161\" retry=\"3\" timeout=\"800\" read-community=\"public\" version=\"v1\" max-repetitions=\"17\" max-vars-per-pdu=\"13\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\"/>\n";
    	
    	final String expectedConfig = 
		"<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
		"<snmp-config port=\"161\" retry=\"3\" timeout=\"800\" read-community=\"public\" version=\"v1\" max-repetitions=\"17\" max-vars-per-pdu=\"13\" xmlns=\"http://xmlns.opennms.org/xsd/config/snmp\">\n"+
		"    <definition proxy-host=\"127.0.0.1\">\n" + 
		"        <specific>192.168.0.8</specific>\n" + 
		"    </definition>\n" + 
		"</snmp-config>\n";
    	

        SnmpPeerFactory.setInstance(new SnmpPeerFactory(new StringResource(snmpConfigXml)));
        assertXmlEquals(snmpConfigXml, SnmpPeerFactory.getInstance().getSnmpConfigAsString());

        SnmpEventInfo info = new SnmpEventInfo();
        info.setProxyHost("127.0.0.1");
        info.setFirstIPAddress("192.168.0.8");
        
        SnmpPeerFactory.getInstance().define(info);
        
        String actualConfig = SnmpPeerFactory.getInstance().getSnmpConfigAsString();
        assertXmlEquals(expectedConfig, actualConfig);
    }
    
	/**
	 * Tests that a SnmpEventInfo creates an event with all parameters. It also
	 * tests that from a given event a SnmpEventInfo object can be created.
	 * @throws UnknownHostException 
	 */
    @Test
	public void testCreateFromEvent() throws UnknownHostException {
		// create an input event. Each attribute must have a different value!
		SnmpEventInfo initial = new SnmpEventInfo();
		initial.setAuthPassPhrase("authPassPhrase");
		initial.setAuthProtocol("authProtocol");
		initial.setContextEngineId("contextEngineId");
		initial.setContextName("contextName");
		initial.setEngineId("engineId");
		initial.setEnterpriseId("enterpriseId");
		initial.setFirstIPAddress("1.1.1.1");
		initial.setLastIPAddress("1.1.1.9");
		initial.setMaxRepetitions(1000);
		initial.setMaxRequestSize(2000);
		initial.setMaxVarsPerPdu(3000);
		initial.setPort(4000);
		initial.setPrivPassPhrase("privPassPhrase");
		initial.setPrivProtocol("privProtocol");
		initial.setProxyHost("proxyHost");
		initial.setReadCommunityString("readCommunityString");
		initial.setRetryCount(5000);
		initial.setSecurityLevel(6000);
		initial.setSecurityName("securityName");
		initial.setTimeout(7000);
		initial.setVersion("version");
		initial.setWriteCommunityString("writeCommunityString");

		// create an event from object and test mapping
		Event event = initial.createEvent("anySource");
		assertEquals(EventConstants.CONFIGURE_SNMP_EVENT_UEI, event.getUei());
		assertEquals(initial.getFirstIPAddress(), event.getInterface());
		assertTrue("Service is not set", event.getService() != null);
		
		// expected values
		Map<String, String> expectedParmMap = new HashMap<String, String>();
		expectedParmMap.put(EventConstants.PARM_FIRST_IP_ADDRESS, initial.getFirstIPAddress());
		expectedParmMap.put(EventConstants.PARM_LAST_IP_ADDRESS, initial.getLastIPAddress());
	    expectedParmMap.put(EventConstants.PARM_SNMP_AUTH_PASSPHRASE, initial.getAuthPassphrase());
	    expectedParmMap.put(EventConstants.PARM_SNMP_AUTH_PROTOCOL, initial.getAuthProtocol());
	    expectedParmMap.put(EventConstants.PARM_SNMP_CONTEXT_ENGINE_ID, initial.getContextEngineId());
	    expectedParmMap.put(EventConstants.PARM_SNMP_CONTEXT_NAME, initial.getContextName());
	    expectedParmMap.put(EventConstants.PARM_SNMP_ENGINE_ID, initial.getEngineId());
	    expectedParmMap.put(EventConstants.PARM_SNMP_ENTERPRISE_ID, initial.getEnterpriseId());
	    expectedParmMap.put(EventConstants.PARM_SNMP_MAX_REPETITIONS, Integer.toString(initial.getMaxRepetitions()));
	    expectedParmMap.put(EventConstants.PARM_SNMP_MAX_REQUEST_SIZE, Integer.toString(initial.getMaxRequestSize()));
	    expectedParmMap.put(EventConstants.PARM_SNMP_MAX_VARS_PER_PDU, Integer.toString(initial.getMaxVarsPerPdu()));
	    expectedParmMap.put(EventConstants.PARM_PORT, Integer.toString(initial.getPort()));
	    expectedParmMap.put(EventConstants.PARM_SNMP_PRIVACY_PASSPHRASE, initial.getPrivPassPhrase());
	    expectedParmMap.put(EventConstants.PARM_SNMP_PRIVACY_PROTOCOL, initial.getPrivProtocol());
	    expectedParmMap.put(EventConstants.PARM_SNMP_PROXY_HOST, initial.getProxyHost());
	    expectedParmMap.put(EventConstants.PARM_SNMP_READ_COMMUNITY_STRING, initial.getReadCommunityString());
	    expectedParmMap.put(EventConstants.PARM_SNMP_SECURITY_NAME, initial.getSecurityName());
	    expectedParmMap.put(EventConstants.PARM_RETRY_COUNT, Integer.toString(initial.getRetryCount()));
	    expectedParmMap.put(EventConstants.PARM_SNMP_SECURITY_LEVEL, Integer.toString(initial.getSecurityLevel()));
	    expectedParmMap.put(EventConstants.PARM_TIMEOUT, Integer.toString(initial.getTimeout()));
	    expectedParmMap.put(EventConstants.PARM_VERSION, initial.getVersion());
	    expectedParmMap.put(EventConstants.PARM_SNMP_WRITE_COMMUNITY_STRING, initial.getWriteCommunityString());
	    
		// ... check each event param, it must be equal to initial
		for (Parm eachParm : event.getParmCollection()) {
			Object expectedValue = expectedParmMap.get(eachParm.getParmName());
			if (expectedValue == null) fail("expectedValue must not be null. Mapping is not implemented correctly");
			assertEquals(expectedValue, eachParm.getValue().getContent());
		}

		// now map the event back to an SnmpEventInfo-object ...
		// ... and check that second is equally to initial
		SnmpEventInfo second = new SnmpEventInfo(ImmutableMapper.fromMutableEvent(event));
		assertEquals(initial, second);
	}
}
