/*
 * Decompiled with CFR 0.152.
 */
package org.apache.curator.framework.imps;

import com.google.common.annotations.VisibleForTesting;
import java.io.EOFException;
import java.io.IOException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.zip.CRC32;
import java.util.zip.DataFormatException;
import java.util.zip.Deflater;
import java.util.zip.Inflater;
import java.util.zip.ZipException;
import org.apache.curator.framework.api.CompressionProvider;

public class GzipCompressionProvider
implements CompressionProvider {
    private static final int MAX_SAFE_JAVA_BYTE_ARRAY_SIZE = 0x7FFFFF7F;
    private static final int GZIP_MAGIC = 35615;
    private static final byte OS_BIT;
    private static final byte[] GZIP_HEADER;
    private static final int FHCRC = 2;
    private static final int FEXTRA = 4;
    private static final int FNAME = 8;
    private static final int FCOMMENT = 16;
    private static final int GZIP_HEADER_SIZE;
    private static final int GZIP_TRAILER_SIZE = 8;
    private static final int MIN_COMPRESSED_DATA_SIZE = 2;
    private static final ConcurrentLinkedQueue<Deflater> DEFLATER_POOL;
    private static final ConcurrentLinkedQueue<Inflater> INFLATER_POOL;
    private static final byte[] COMPRESSED_EMPTY_BYTES;

    private static Deflater acquireDeflater() {
        Deflater deflater = DEFLATER_POOL.poll();
        if (deflater == null) {
            deflater = new Deflater(-1, true);
        }
        return deflater;
    }

    private static Inflater acquireInflater() {
        Inflater inflater = INFLATER_POOL.poll();
        if (inflater == null) {
            inflater = new Inflater(true);
        }
        return inflater;
    }

    @Override
    public byte[] compress(String path, byte[] data) {
        if (data.length == 0) {
            return (byte[])COMPRESSED_EMPTY_BYTES.clone();
        }
        return GzipCompressionProvider.doCompress(data);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    static byte[] doCompress(byte[] data) {
        byte[] result = Arrays.copyOf(GZIP_HEADER, GzipCompressionProvider.conservativeGZippedSizeEstimate(data.length));
        Deflater deflater = GzipCompressionProvider.acquireDeflater();
        try {
            deflater.setInput(data);
            deflater.finish();
            int offset = GZIP_HEADER_SIZE;
            while (true) {
                int available = result.length - 8 - offset;
                int numCompressedBytes = deflater.deflate(result, offset, available);
                offset += numCompressedBytes;
                if (deflater.finished()) break;
                int newResultLength = result.length + result.length / 2;
                result = Arrays.copyOf(result, newResultLength);
            }
            CRC32 crc = new CRC32();
            crc.update(data, 0, data.length);
            GzipCompressionProvider.writeLittleEndianInt(result, offset, (int)crc.getValue());
            GzipCompressionProvider.writeLittleEndianInt(result, offset + 4, data.length);
            int endOffset = offset + 8;
            if (result.length != endOffset) {
                result = Arrays.copyOf(result, endOffset);
            }
            byte[] byArray = result;
            return byArray;
        }
        finally {
            deflater.reset();
            DEFLATER_POOL.add(deflater);
        }
    }

    private static int conservativeGZippedSizeEstimate(int dataSize) {
        int conservativeCompressedDataSizeEstimate = dataSize < 512 ? Math.max(dataSize, 2) : Math.max(512, dataSize / 2);
        return GZIP_HEADER_SIZE + conservativeCompressedDataSizeEstimate + 8;
    }

    private static void writeLittleEndianInt(byte[] b, int offset, int v) {
        b[offset] = (byte)v;
        b[offset + 1] = (byte)(v >> 8);
        b[offset + 2] = (byte)(v >> 16);
        b[offset + 3] = (byte)(v >> 24);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public byte[] decompress(String path, byte[] gzippedDataBytes) throws IOException {
        if (Arrays.equals(gzippedDataBytes, COMPRESSED_EMPTY_BYTES)) {
            return new byte[0];
        }
        ByteBuffer gzippedData = ByteBuffer.wrap(gzippedDataBytes);
        gzippedData.order(ByteOrder.LITTLE_ENDIAN);
        int headerSize = GzipCompressionProvider.readGzipHeader(gzippedData);
        if (gzippedDataBytes.length < headerSize + 2 + 8) {
            throw new EOFException("Too short GZipped data");
        }
        int compressedDataSize = gzippedDataBytes.length - headerSize - 8;
        int initialResultLength = (int)Math.min((long)compressedDataSize * 3L, 0x7FFFFF7FL);
        byte[] result = new byte[initialResultLength];
        Inflater inflater = GzipCompressionProvider.acquireInflater();
        try {
            inflater.setInput(gzippedDataBytes, headerSize, compressedDataSize);
            CRC32 crc = new CRC32();
            int offset = 0;
            while (true) {
                int numDecompressedBytes;
                try {
                    numDecompressedBytes = inflater.inflate(result, offset, result.length - offset);
                }
                catch (DataFormatException e) {
                    String s = e.getMessage();
                    throw new ZipException(s != null ? s : "Invalid ZLIB data format");
                }
                crc.update(result, offset, numDecompressedBytes);
                offset += numDecompressedBytes;
                if (inflater.finished() || inflater.needsDictionary()) break;
                if (numDecompressedBytes == 0 && inflater.needsInput()) {
                    throw new ZipException("Corrupt GZipped data");
                }
                if (result.length == 0x7FFFFF7F && numDecompressedBytes == 0) {
                    throw new OutOfMemoryError("Unable to uncompress that much data into a single byte[] array");
                }
                int newResultLength = (int)Math.min((long)result.length + (long)(result.length / 2), 0x7FFFFF7FL);
                if (result.length == newResultLength) continue;
                result = Arrays.copyOf(result, newResultLength);
            }
            if (inflater.getRemaining() != 0) {
                throw new ZipException("Expected just one GZip block, without garbage in the end");
            }
            int checksum = gzippedData.getInt(gzippedDataBytes.length - 8);
            int numUncompressedBytes = gzippedData.getInt(gzippedDataBytes.length - 4);
            if (checksum != (int)crc.getValue() || numUncompressedBytes != offset) {
                throw new ZipException("Corrupt GZIP trailer");
            }
            if (result.length != offset) {
                result = Arrays.copyOf(result, offset);
            }
            byte[] byArray = result;
            return byArray;
        }
        finally {
            inflater.reset();
            INFLATER_POOL.add(inflater);
        }
    }

    private static int readGzipHeader(ByteBuffer gzippedData) throws IOException {
        try {
            return GzipCompressionProvider.doReadHeader(gzippedData);
        }
        catch (BufferUnderflowException e) {
            throw new EOFException();
        }
    }

    private static int doReadHeader(ByteBuffer gzippedData) throws IOException {
        if (gzippedData.getChar() != '\u8b1f') {
            throw new ZipException("Not in GZip format");
        }
        if (gzippedData.get() != 8) {
            throw new ZipException("Unsupported compression method");
        }
        byte flags = gzippedData.get();
        GzipCompressionProvider.skip(gzippedData, 6);
        if ((flags & 4) != 0) {
            char extraBytes = gzippedData.getChar();
            GzipCompressionProvider.skip(gzippedData, extraBytes);
        }
        if ((flags & 8) != 0) {
            GzipCompressionProvider.skipZeroTerminatedString(gzippedData);
        }
        if ((flags & 0x10) != 0) {
            GzipCompressionProvider.skipZeroTerminatedString(gzippedData);
        }
        if ((flags & 2) != 0) {
            CRC32 crc = new CRC32();
            crc.update(gzippedData.array(), 0, gzippedData.position());
            if (gzippedData.getChar() != (char)crc.getValue()) {
                throw new ZipException("Corrupt GZIP header");
            }
        }
        return gzippedData.position();
    }

    private static void skip(ByteBuffer gzippedData, int skipBytes) throws IOException {
        try {
            gzippedData.position(gzippedData.position() + skipBytes);
        }
        catch (IllegalArgumentException e) {
            throw new EOFException();
        }
    }

    private static void skipZeroTerminatedString(ByteBuffer gzippedData) {
        while (gzippedData.get() != 0) {
        }
    }

    static {
        String version = System.getProperty("java.specification.version");
        OS_BIT = version.contains(".") ? (byte)0 : (byte)(Float.parseFloat(version) >= 16.0f ? -1 : 0);
        GZIP_HEADER = new byte[]{31, -117, 8, 0, 0, 0, 0, 0, 0, OS_BIT};
        GZIP_HEADER_SIZE = GZIP_HEADER.length;
        DEFLATER_POOL = new ConcurrentLinkedQueue();
        INFLATER_POOL = new ConcurrentLinkedQueue();
        COMPRESSED_EMPTY_BYTES = new byte[]{31, -117, 8, 0, 0, 0, 0, 0, 0, OS_BIT, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    }
}

