/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.config.poller;

import java.io.Serializable;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlRootElement;

/**
 * Critical service. Defining a critical service greatly reduces the traffic
 * generated by the poller when an interface is DOWN. When an interface is
 * DOWN only the critical service is polled. If and when the critical service
 * comes back UP then the interface's other services are polled to determine
 * their status. When an interface is UP all services are polled as expected.
 * If the critical service goes DOWN, all services are considered to be DOWN
 * and therefore the interface is also considered DOWN.
 */

@XmlRootElement(name="critical-service")
@XmlAccessorType(XmlAccessType.NONE)
public class CriticalService implements Serializable {
    private static final long serialVersionUID = 3608855241271932451L;

    /**
     * Name of the critical service (typically, "ICMP")
     */
    @XmlAttribute(name="name")
    private String m_name;


    public CriticalService() {
        super();
    }

    public CriticalService(final String name) {
        this();
        setName(name);
    }

    /**
     * Name of the critical service (typically, "ICMP")
     */
    public String getName() {
        return m_name;
    }

    public void setName(final String name) {
        m_name = name;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((m_name == null) ? 0 : m_name.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CriticalService)) {
            return false;
        }
        CriticalService other = (CriticalService) obj;
        if (m_name == null) {
            if (other.m_name != null) {
                return false;
            }
        } else if (!m_name.equals(other.m_name)) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        return "CriticalService[" + m_name + "]";
    }
}
