/*
 * Licensed to The OpenNMS Group, Inc (TOG) under one or more
 * contributor license agreements.  See the LICENSE.md file
 * distributed with this work for additional information
 * regarding copyright ownership.
 *
 * TOG licenses this file to You under the GNU Affero General
 * Public License Version 3 (the "License") or (at your option)
 * any later version.  You may not use this file except in
 * compliance with the License.  You may obtain a copy of the
 * License at:
 *
 *      https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.  See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package org.opennms.netmgt.timeseries.resource;

import static org.opennms.netmgt.timeseries.util.TimeseriesUtils.toMetricName;
import static org.opennms.netmgt.timeseries.util.TimeseriesUtils.toResourcePath;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.opennms.integration.api.v1.timeseries.IntrinsicTagNames;
import org.opennms.integration.api.v1.timeseries.MetaTagNames;
import org.opennms.integration.api.v1.timeseries.Metric;
import org.opennms.integration.api.v1.timeseries.StorageException;
import org.opennms.integration.api.v1.timeseries.Tag;
import org.opennms.netmgt.dao.api.ResourceStorageDao;
import org.opennms.netmgt.model.OnmsAttribute;
import org.opennms.netmgt.model.ResourcePath;
import org.opennms.netmgt.model.ResourceTypeUtils;
import org.opennms.netmgt.model.RrdGraphAttribute;
import org.opennms.netmgt.model.StringPropertyAttribute;
import org.opennms.netmgt.timeseries.TimeseriesStorageManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;

/**
 * Resource Storage Dao implementation for Timeseries Integration Layer that leverages the Search API for walking the resource tree.
 *
 * In Timeseries Integration Layer, samples are associated with metrics, which are in turn associated with resources.
 *
 * Here we split the resource id into two parts:
 *   bucket: last element of the resource id
 *   resource path: all the elements before the bucket
 * Relating this to .rrd file on disk, the bucket would be the filename, and the resource path would be its folder.
 *
 */
public class TimeseriesResourceStorageDao implements ResourceStorageDao {

    private static final Logger LOG = LoggerFactory.getLogger(TimeseriesResourceStorageDao.class);

    @Autowired
    private TimeseriesStorageManager storageManager;

    @Autowired
    private TimeseriesSearcher searcher;

    @Override
    public boolean exists(ResourcePath path, int depth) {
        Preconditions.checkArgument(depth >= 0, "depth must be non-negative");
        return searchFor(path, depth).size() > 0;
    }

    @Override
    public boolean existsWithin(final ResourcePath path, final int depth) {
        Preconditions.checkArgument(depth >= 0, "depth must be non-negative");

        // The indices are structured in such a way that we need specify the depth
        // so here we need to iterate over all the possibilities. We could add
        // additional indices to avoid this, but it's not worth the additional
        // writes, since the specified depth should be relatively small.
        return IntStream.rangeClosed(0, depth)
            .anyMatch(i -> searchFor(path, i).size() > 0);
    }

    @Override
    public Set<ResourcePath> children(ResourcePath path, int depth) {
        Preconditions.checkArgument(depth >= 0, "depth must be non-negative");
        Set<ResourcePath> matches = Sets.newTreeSet();

        Set<Metric> metrics = searchFor(path, depth);
        for (Metric metric : metrics) {
            // Relativize the path
            ResourcePath child = toChildResourcePath(path, metric.getFirstTagByKey(IntrinsicTagNames.resourceId).getValue());
            if (child == null) {
                // This shouldn't happen
                LOG.warn("Encountered non-child resource {} when searching for {} with depth {}. Ignoring resource.",
                        metric.getFirstTagByKey(IntrinsicTagNames.resourceId).getValue(), path, depth);
                continue;
            }
            matches.add(child);
        }
        return matches;
    }

    @Override
    public boolean delete(ResourcePath path) {
        final Set<Metric> results = searchFor(path, 0);

        if (results.isEmpty()) {
            return false;
        }

        for (final Metric metric : results) {
                try {
                    storageManager.get().delete(metric);
                } catch (StorageException e) {
                    LOG.error("Could not delete {}, will ignore problem and continue ", metric, e);
                }
        }

        return true;
    }

    @Override
    public Set<OnmsAttribute> getAttributes(ResourcePath path) {
        Set<OnmsAttribute> attributes = Sets.newHashSet();

        // Gather the list of metrics available under the resource path
        Set<Metric> metrics = searchFor(path, 0);
        for (Metric metric : metrics) {
            final String resourceId = metric.getFirstTagByKey(IntrinsicTagNames.resourceId).getValue();
            final ResourcePath resultPath = toResourcePath(resourceId);
            if (!path.equals(resultPath)) {
                // The paths don't match exactly, but it is possible that they differ only by leading/trailing whitespace
                // so we perform a closer inspection
                if (!Arrays.stream(path.elements())
                        .map(String::trim)
                        .collect(Collectors.toList())
                        .equals(Arrays.asList(resultPath.elements()))) {
                    // This shouldn't happen
                    LOG.warn("Encountered non-child resource {} when searching for {} with depth {}. " +
                            "Ignoring resource.", resourceId, path, 0);
                    continue;
                }
            }

            if (ResourceTypeUtils.isResponseTime(resourceId) || ResourceTypeUtils.isStatus(resourceId)) {
                // Use the last part of the resource id as the dsName
                // Store the resource id in the rrdFile field
                attributes.add(new RrdGraphAttribute(toMetricName(resourceId), "", resourceId));
            } else {
                // Use the metric name as the dsName
                // Store the resource id in the rrdFile field
                attributes.add(new RrdGraphAttribute(metric.getFirstTagByKey(IntrinsicTagNames.name).getValue(), "", resourceId));
            }
        }

        // Add the resource level attributes to the result set
        Set<Metric> metricsWithStringAttributes = new HashSet<>(metrics);
        metricsWithStringAttributes.addAll(searchFor(path, -1));
        if (!metricsWithStringAttributes.isEmpty()) {
            metricsWithStringAttributes.iterator().next()
                    .getExternalTags().stream()
                    .map(t -> new StringPropertyAttribute(t.getKey(), t.getValue()))
                    .forEach(attributes::add);
        }
        return attributes;
    }

    @Override
    public void setStringAttribute(ResourcePath path, String key, String value) {
        throw new UnsupportedOperationException("This method is not supported anymore. Please use KV store instead.");
    }

    @Override
    public String getStringAttribute(ResourcePath path, String key) {
        return getStringAttributes(path).get(key);
    }

    @Override
    public Map<String, String> getStringAttributes(ResourcePath path) {
        return getMetaData(path);
    }

    @Override
    public Map<String, String> getMetaData(ResourcePath path) {
        Set<Metric> metricsWithStringAttributes = new HashSet<>();
        metricsWithStringAttributes.addAll(searchFor(path, 0));
        metricsWithStringAttributes.addAll(searchFor(path, -1)); // resource level
        return metricsWithStringAttributes
                    .stream()
                    .flatMap(m -> m.getExternalTags().stream())
                    .distinct()
                    .filter(t -> !t.getKey().endsWith(MetaTagNames.mtype)) // mtype is on Metric level => we are one above (resource level)
                    .collect(Collectors.toMap(Tag::getKey, Tag::getValue));
    }

    @Override
    public void updateMetricToResourceMappings(ResourcePath path, Map<String, String> metricsNameToResourceNames) {
        // These are already stored.
    }

    private Set<Metric> searchFor(ResourcePath path, int depth) {
        final Set<Metric> results;
        try {
            results = searcher.search(path, depth);
            LOG.trace("Found {} results.", results.size());
        } catch (StorageException e) {
            LOG.error("An error occurred while querying for {}", path, e);
            throw new RuntimeException(e);
        }
        return results;
    }

    protected static ResourcePath toChildResourcePath(ResourcePath parent, String resourceId) {
        final ResourcePath child = toResourcePath(resourceId);
        final String childEls[] = child.elements();
        final String parentEls[] = parent.elements();

        if (childEls == null || parentEls == null || childEls.length <= parentEls.length) {
            return null;
        }

        String els[] = new String[parentEls.length + 1];
        for (int i = 0; i <= parentEls.length ; i++) {
            els[i] = childEls[i];
        }

        return ResourcePath.get(els);
    }

    public void setSearcher(TimeseriesSearcher searcher) {
        this.searcher = searcher;
    }
}
